# This test verifies that if there's a series of rw-conflicts
#   s1 ---> s2 ---> s3, with s3 committing first
# at least one transaction will be aborted, regardless of the order in
# which the conflicts are detected and transactions prepare and
# commit.
#
#
# Tables test2 and test3 are used to create the
# s1 --> s2 and s2 --> s3 rw-dependencies respectively
#
# test1 isn't involved in the anomaly; s1 only inserts a row into it
# so that there's an easy way to tell (by looking for that row) if s1
# successfully committed.
#
# force_snapshot is used to force s2 and s3 to take their snapshot
# immediately after BEGIN, so we can be sure the three transactions
# overlap.
setup
{
    CREATE TABLE test1 (a int);
    CREATE TABLE test2 (b int);
    CREATE TABLE test3 (c int);
    CREATE TABLE force_snapshot (a int);
}

teardown
{
    DROP TABLE test1;
    DROP TABLE test2;
    DROP TABLE test3;
    DROP TABLE force_snapshot;
}


session s1
setup { BEGIN ISOLATION LEVEL SERIALIZABLE; INSERT INTO test1 VALUES (1); }
step r1 { SELECT * FROM test2; }
step p1 { PREPARE TRANSACTION 's1'; }
step c1 { COMMIT PREPARED 's1'; }

session s2
setup { BEGIN ISOLATION LEVEL SERIALIZABLE; SELECT * FROM force_snapshot; }
step r2 { SELECT * FROM test3; }
step w2 { INSERT INTO test2 VALUES (2); }
step p2 { PREPARE TRANSACTION 's2'; }
step c2 { COMMIT PREPARED 's2'; }

session s3
setup { BEGIN ISOLATION LEVEL SERIALIZABLE; SELECT * FROM force_snapshot; }
step w3 { INSERT INTO test3 VALUES (3); }
step p3 { PREPARE TRANSACTION 's3'; }
step c3 { COMMIT PREPARED 's3'; }

# When run at the end of the permutations below, this SELECT statement
# should never return any tuples, because at least one of the three
# transactions involved should be aborted.
session s4
step check { SELECT * FROM test1,test2,test3; }

# We run on all permutations of the statements above subject to the
# following constraints:
#   - each transaction's reads/writes must happen before it prepares
#   - each transaction must prepare before committing
#   - s3 must be the first of the three transactions to commit
# ...which means that each permutation should fail.
#
# To bring the number of permutations down a bit, we further require
# them to satisfy *one* of the following:
#  - r1 < w2 < r2 < w3   (T1-->T2 conflict on write, T2-->T3 on write)
#  - r1 < w2 < w3 < r2   (T1-->T2 conflict on read, T2-->T3 on write)
#  - w2 < r1 < r2 < w3   (T1-->T2 conflict on write, T2-->T3 on read)
#  - w2 < r1 < w3 < r2   (T1-->T2 conflict on read, T2-->T3 on read)
# This eliminates some redundant combinations. For example, it doesn't
# matter if w2 happens before w3 as long as both come before the
# conflicting reads.
permutation  r1  r2  w2  w3  p1  p2  p3  c3  c1  c2  check
permutation  r1  r2  w2  w3  p1  p2  p3  c3  c2  c1  check
permutation  r1  r2  w2  w3  p1  p3  p2  c3  c1  c2  check
permutation  r1  r2  w2  w3  p1  p3  p2  c3  c2  c1  check
permutation  r1  r2  w2  w3  p1  p3  c3  p2  c1  c2  check
permutation  r1  r2  w2  w3  p1  p3  c3  p2  c2  c1  check
permutation  r1  r2  w2  w3  p1  p3  c3  c1  p2  c2  check
permutation  r1  r2  w2  w3  p2  p1  p3  c3  c1  c2  check
permutation  r1  r2  w2  w3  p2  p1  p3  c3  c2  c1  check
permutation  r1  r2  w2  w3  p2  p3  p1  c3  c1  c2  check
permutation  r1  r2  w2  w3  p2  p3  p1  c3  c2  c1  check
permutation  r1  r2  w2  w3  p2  p3  c3  p1  c1  c2  check
permutation  r1  r2  w2  w3  p2  p3  c3  p1  c2  c1  check
permutation  r1  r2  w2  w3  p2  p3  c3  c2  p1  c1  check
permutation  r1  r2  w2  w3  p3  p1  p2  c3  c1  c2  check
permutation  r1  r2  w2  w3  p3  p1  p2  c3  c2  c1  check
permutation  r1  r2  w2  w3  p3  p1  c3  p2  c1  c2  check
permutation  r1  r2  w2  w3  p3  p1  c3  p2  c2  c1  check
permutation  r1  r2  w2  w3  p3  p1  c3  c1  p2  c2  check
permutation  r1  r2  w2  w3  p3  p2  p1  c3  c1  c2  check
permutation  r1  r2  w2  w3  p3  p2  p1  c3  c2  c1  check
permutation  r1  r2  w2  w3  p3  p2  c3  p1  c1  c2  check
permutation  r1  r2  w2  w3  p3  p2  c3  p1  c2  c1  check
permutation  r1  r2  w2  w3  p3  p2  c3  c2  p1  c1  check
permutation  r1  r2  w2  w3  p3  c3  p1  p2  c1  c2  check
permutation  r1  r2  w2  w3  p3  c3  p1  p2  c2  c1  check
permutation  r1  r2  w2  w3  p3  c3  p1  c1  p2  c2  check
permutation  r1  r2  w2  w3  p3  c3  p2  p1  c1  c2  check
permutation  r1  r2  w2  w3  p3  c3  p2  p1  c2  c1  check
permutation  r1  r2  w2  w3  p3  c3  p2  c2  p1  c1  check
permutation  r1  r2  w2  p1  w3  p2  p3  c3  c1  c2  check
permutation  r1  r2  w2  p1  w3  p2  p3  c3  c2  c1  check
permutation  r1  r2  w2  p1  w3  p3  p2  c3  c1  c2  check
permutation  r1  r2  w2  p1  w3  p3  p2  c3  c2  c1  check
permutation  r1  r2  w2  p1  w3  p3  c3  p2  c1  c2  check
permutation  r1  r2  w2  p1  w3  p3  c3  p2  c2  c1  check
permutation  r1  r2  w2  p1  w3  p3  c3  c1  p2  c2  check
permutation  r1  r2  w2  p1  p2  w3  p3  c3  c1  c2  check
permutation  r1  r2  w2  p1  p2  w3  p3  c3  c2  c1  check
permutation  r1  r2  w2  p2  w3  p1  p3  c3  c1  c2  check
permutation  r1  r2  w2  p2  w3  p1  p3  c3  c2  c1  check
permutation  r1  r2  w2  p2  w3  p3  p1  c3  c1  c2  check
permutation  r1  r2  w2  p2  w3  p3  p1  c3  c2  c1  check
permutation  r1  r2  w2  p2  w3  p3  c3  p1  c1  c2  check
permutation  r1  r2  w2  p2  w3  p3  c3  p1  c2  c1  check
permutation  r1  r2  w2  p2  w3  p3  c3  c2  p1  c1  check
permutation  r1  r2  w2  p2  p1  w3  p3  c3  c1  c2  check
permutation  r1  r2  w2  p2  p1  w3  p3  c3  c2  c1  check
permutation  r1  r2  p1  w2  w3  p2  p3  c3  c1  c2  check
permutation  r1  r2  p1  w2  w3  p2  p3  c3  c2  c1  check
permutation  r1  r2  p1  w2  w3  p3  p2  c3  c1  c2  check
permutation  r1  r2  p1  w2  w3  p3  p2  c3  c2  c1  check
permutation  r1  r2  p1  w2  w3  p3  c3  p2  c1  c2  check
permutation  r1  r2  p1  w2  w3  p3  c3  p2  c2  c1  check
permutation  r1  r2  p1  w2  w3  p3  c3  c1  p2  c2  check
permutation  r1  r2  p1  w2  p2  w3  p3  c3  c1  c2  check
permutation  r1  r2  p1  w2  p2  w3  p3  c3  c2  c1  check
permutation  r1  w2  w3  r2  p1  p2  p3  c3  c1  c2  check
permutation  r1  w2  w3  r2  p1  p2  p3  c3  c2  c1  check
permutation  r1  w2  w3  r2  p1  p3  p2  c3  c1  c2  check
permutation  r1  w2  w3  r2  p1  p3  p2  c3  c2  c1  check
permutation  r1  w2  w3  r2  p1  p3  c3  p2  c1  c2  check
permutation  r1  w2  w3  r2  p1  p3  c3  p2  c2  c1  check
permutation  r1  w2  w3  r2  p1  p3  c3  c1  p2  c2  check
permutation  r1  w2  w3  r2  p2  p1  p3  c3  c1  c2  check
permutation  r1  w2  w3  r2  p2  p1  p3  c3  c2  c1  check
permutation  r1  w2  w3  r2  p2  p3  p1  c3  c1  c2  check
permutation  r1  w2  w3  r2  p2  p3  p1  c3  c2  c1  check
permutation  r1  w2  w3  r2  p2  p3  c3  p1  c1  c2  check
permutation  r1  w2  w3  r2  p2  p3  c3  p1  c2  c1  check
permutation  r1  w2  w3  r2  p2  p3  c3  c2  p1  c1  check
permutation  r1  w2  w3  r2  p3  p1  p2  c3  c1  c2  check
permutation  r1  w2  w3  r2  p3  p1  p2  c3  c2  c1  check
permutation  r1  w2  w3  r2  p3  p1  c3  p2  c1  c2  check
permutation  r1  w2  w3  r2  p3  p1  c3  p2  c2  c1  check
permutation  r1  w2  w3  r2  p3  p1  c3  c1  p2  c2  check
permutation  r1  w2  w3  r2  p3  p2  p1  c3  c1  c2  check
permutation  r1  w2  w3  r2  p3  p2  p1  c3  c2  c1  check
permutation  r1  w2  w3  r2  p3  p2  c3  p1  c1  c2  check
permutation  r1  w2  w3  r2  p3  p2  c3  p1  c2  c1  check
permutation  r1  w2  w3  r2  p3  p2  c3  c2  p1  c1  check
permutation  r1  w2  w3  r2  p3  c3  p1  p2  c1  c2  check
permutation  r1  w2  w3  r2  p3  c3  p1  p2  c2  c1  check
permutation  r1  w2  w3  r2  p3  c3  p1  c1  p2  c2  check
permutation  r1  w2  w3  r2  p3  c3  p2  p1  c1  c2  check
permutation  r1  w2  w3  r2  p3  c3  p2  p1  c2  c1  check
permutation  r1  w2  w3  r2  p3  c3  p2  c2  p1  c1  check
permutation  r1  w2  w3  p1  r2  p2  p3  c3  c1  c2  check
permutation  r1  w2  w3  p1  r2  p2  p3  c3  c2  c1  check
permutation  r1  w2  w3  p1  r2  p3  p2  c3  c1  c2  check
permutation  r1  w2  w3  p1  r2  p3  p2  c3  c2  c1  check
permutation  r1  w2  w3  p1  r2  p3  c3  p2  c1  c2  check
permutation  r1  w2  w3  p1  r2  p3  c3  p2  c2  c1  check
permutation  r1  w2  w3  p1  r2  p3  c3  c1  p2  c2  check
permutation  r1  w2  w3  p1  p3  r2  p2  c3  c1  c2  check
permutation  r1  w2  w3  p1  p3  r2  p2  c3  c2  c1  check
permutation  r1  w2  w3  p1  p3  r2  c3  p2  c1  c2  check
permutation  r1  w2  w3  p1  p3  r2  c3  p2  c2  c1  check
permutation  r1  w2  w3  p1  p3  r2  c3  c1  p2  c2  check
permutation  r1  w2  w3  p1  p3  c3  r2  p2  c1  c2  check
permutation  r1  w2  w3  p1  p3  c3  r2  p2  c2  c1  check
permutation  r1  w2  w3  p1  p3  c3  r2  c1  p2  c2  check
permutation  r1  w2  w3  p1  p3  c3  c1  r2  p2  c2  check
permutation  r1  w2  w3  p3  r2  p1  p2  c3  c1  c2  check
permutation  r1  w2  w3  p3  r2  p1  p2  c3  c2  c1  check
permutation  r1  w2  w3  p3  r2  p1  c3  p2  c1  c2  check
permutation  r1  w2  w3  p3  r2  p1  c3  p2  c2  c1  check
permutation  r1  w2  w3  p3  r2  p1  c3  c1  p2  c2  check
permutation  r1  w2  w3  p3  r2  p2  p1  c3  c1  c2  check
permutation  r1  w2  w3  p3  r2  p2  p1  c3  c2  c1  check
permutation  r1  w2  w3  p3  r2  p2  c3  p1  c1  c2  check
permutation  r1  w2  w3  p3  r2  p2  c3  p1  c2  c1  check
permutation  r1  w2  w3  p3  r2  p2  c3  c2  p1  c1  check
permutation  r1  w2  w3  p3  r2  c3  p1  p2  c1  c2  check
permutation  r1  w2  w3  p3  r2  c3  p1  p2  c2  c1  check
permutation  r1  w2  w3  p3  r2  c3  p1  c1  p2  c2  check
permutation  r1  w2  w3  p3  r2  c3  p2  p1  c1  c2  check
permutation  r1  w2  w3  p3  r2  c3  p2  p1  c2  c1  check
permutation  r1  w2  w3  p3  r2  c3  p2  c2  p1  c1  check
permutation  r1  w2  w3  p3  p1  r2  p2  c3  c1  c2  check
permutation  r1  w2  w3  p3  p1  r2  p2  c3  c2  c1  check
permutation  r1  w2  w3  p3  p1  r2  c3  p2  c1  c2  check
permutation  r1  w2  w3  p3  p1  r2  c3  p2  c2  c1  check
permutation  r1  w2  w3  p3  p1  r2  c3  c1  p2  c2  check
permutation  r1  w2  w3  p3  p1  c3  r2  p2  c1  c2  check
permutation  r1  w2  w3  p3  p1  c3  r2  p2  c2  c1  check
permutation  r1  w2  w3  p3  p1  c3  r2  c1  p2  c2  check
permutation  r1  w2  w3  p3  p1  c3  c1  r2  p2  c2  check
permutation  r1  w2  w3  p3  c3  r2  p1  p2  c1  c2  check
permutation  r1  w2  w3  p3  c3  r2  p1  p2  c2  c1  check
permutation  r1  w2  w3  p3  c3  r2  p1  c1  p2  c2  check
permutation  r1  w2  w3  p3  c3  r2  p2  p1  c1  c2  check
permutation  r1  w2  w3  p3  c3  r2  p2  p1  c2  c1  check
permutation  r1  w2  w3  p3  c3  r2  p2  c2  p1  c1  check
permutation  r1  w2  w3  p3  c3  p1  r2  p2  c1  c2  check
permutation  r1  w2  w3  p3  c3  p1  r2  p2  c2  c1  check
permutation  r1  w2  w3  p3  c3  p1  r2  c1  p2  c2  check
permutation  r1  w2  w3  p3  c3  p1  c1  r2  p2  c2  check
permutation  r1  w2  p1  w3  r2  p2  p3  c3  c1  c2  check
permutation  r1  w2  p1  w3  r2  p2  p3  c3  c2  c1  check
permutation  r1  w2  p1  w3  r2  p3  p2  c3  c1  c2  check
permutation  r1  w2  p1  w3  r2  p3  p2  c3  c2  c1  check
permutation  r1  w2  p1  w3  r2  p3  c3  p2  c1  c2  check
permutation  r1  w2  p1  w3  r2  p3  c3  p2  c2  c1  check
permutation  r1  w2  p1  w3  r2  p3  c3  c1  p2  c2  check
permutation  r1  w2  p1  w3  p3  r2  p2  c3  c1  c2  check
permutation  r1  w2  p1  w3  p3  r2  p2  c3  c2  c1  check
permutation  r1  w2  p1  w3  p3  r2  c3  p2  c1  c2  check
permutation  r1  w2  p1  w3  p3  r2  c3  p2  c2  c1  check
permutation  r1  w2  p1  w3  p3  r2  c3  c1  p2  c2  check
permutation  r1  w2  p1  w3  p3  c3  r2  p2  c1  c2  check
permutation  r1  w2  p1  w3  p3  c3  r2  p2  c2  c1  check
permutation  r1  w2  p1  w3  p3  c3  r2  c1  p2  c2  check
permutation  r1  w2  p1  w3  p3  c3  c1  r2  p2  c2  check
permutation  r1  w3  r2  w2  p1  p2  p3  c3  c1  c2  check
permutation  r1  w3  r2  w2  p1  p2  p3  c3  c2  c1  check
permutation  r1  w3  r2  w2  p1  p3  p2  c3  c1  c2  check
permutation  r1  w3  r2  w2  p1  p3  p2  c3  c2  c1  check
permutation  r1  w3  r2  w2  p1  p3  c3  p2  c1  c2  check
permutation  r1  w3  r2  w2  p1  p3  c3  p2  c2  c1  check
permutation  r1  w3  r2  w2  p1  p3  c3  c1  p2  c2  check
permutation  r1  w3  r2  w2  p2  p1  p3  c3  c1  c2  check
permutation  r1  w3  r2  w2  p2  p1  p3  c3  c2  c1  check
permutation  r1  w3  r2  w2  p2  p3  p1  c3  c1  c2  check
permutation  r1  w3  r2  w2  p2  p3  p1  c3  c2  c1  check
permutation  r1  w3  r2  w2  p2  p3  c3  p1  c1  c2  check
permutation  r1  w3  r2  w2  p2  p3  c3  p1  c2  c1  check
permutation  r1  w3  r2  w2  p2  p3  c3  c2  p1  c1  check
permutation  r1  w3  r2  w2  p3  p1  p2  c3  c1  c2  check
permutation  r1  w3  r2  w2  p3  p1  p2  c3  c2  c1  check
permutation  r1  w3  r2  w2  p3  p1  c3  p2  c1  c2  check
permutation  r1  w3  r2  w2  p3  p1  c3  p2  c2  c1  check
permutation  r1  w3  r2  w2  p3  p1  c3  c1  p2  c2  check
permutation  r1  w3  r2  w2  p3  p2  p1  c3  c1  c2  check
permutation  r1  w3  r2  w2  p3  p2  p1  c3  c2  c1  check
permutation  r1  w3  r2  w2  p3  p2  c3  p1  c1  c2  check
permutation  r1  w3  r2  w2  p3  p2  c3  p1  c2  c1  check
permutation  r1  w3  r2  w2  p3  p2  c3  c2  p1  c1  check
permutation  r1  w3  r2  w2  p3  c3  p1  p2  c1  c2  check
permutation  r1  w3  r2  w2  p3  c3  p1  p2  c2  c1  check
permutation  r1  w3  r2  w2  p3  c3  p1  c1  p2  c2  check
permutation  r1  w3  r2  w2  p3  c3  p2  p1  c1  c2  check
permutation  r1  w3  r2  w2  p3  c3  p2  p1  c2  c1  check
permutation  r1  w3  r2  w2  p3  c3  p2  c2  p1  c1  check
permutation  r1  w3  r2  p1  w2  p2  p3  c3  c1  c2  check
permutation  r1  w3  r2  p1  w2  p2  p3  c3  c2  c1  check
permutation  r1  w3  r2  p1  w2  p3  p2  c3  c1  c2  check
permutation  r1  w3  r2  p1  w2  p3  p2  c3  c2  c1  check
permutation  r1  w3  r2  p1  w2  p3  c3  p2  c1  c2  check
permutation  r1  w3  r2  p1  w2  p3  c3  p2  c2  c1  check
permutation  r1  w3  r2  p1  w2  p3  c3  c1  p2  c2  check
permutation  r1  w3  r2  p1  p3  w2  p2  c3  c1  c2  check
permutation  r1  w3  r2  p1  p3  w2  p2  c3  c2  c1  check
permutation  r1  w3  r2  p1  p3  w2  c3  p2  c1  c2  check
permutation  r1  w3  r2  p1  p3  w2  c3  p2  c2  c1  check
permutation  r1  w3  r2  p1  p3  w2  c3  c1  p2  c2  check
permutation  r1  w3  r2  p1  p3  c3  w2  p2  c1  c2  check
permutation  r1  w3  r2  p1  p3  c3  w2  p2  c2  c1  check
permutation  r1  w3  r2  p1  p3  c3  w2  c1  p2  c2  check
permutation  r1  w3  r2  p1  p3  c3  c1  w2  p2  c2  check
permutation  r1  w3  r2  p3  w2  p1  p2  c3  c1  c2  check
permutation  r1  w3  r2  p3  w2  p1  p2  c3  c2  c1  check
permutation  r1  w3  r2  p3  w2  p1  c3  p2  c1  c2  check
permutation  r1  w3  r2  p3  w2  p1  c3  p2  c2  c1  check
permutation  r1  w3  r2  p3  w2  p1  c3  c1  p2  c2  check
permutation  r1  w3  r2  p3  w2  p2  p1  c3  c1  c2  check
permutation  r1  w3  r2  p3  w2  p2  p1  c3  c2  c1  check
permutation  r1  w3  r2  p3  w2  p2  c3  p1  c1  c2  check
permutation  r1  w3  r2  p3  w2  p2  c3  p1  c2  c1  check
permutation  r1  w3  r2  p3  w2  p2  c3  c2  p1  c1  check
permutation  r1  w3  r2  p3  w2  c3  p1  p2  c1  c2  check
permutation  r1  w3  r2  p3  w2  c3  p1  p2  c2  c1  check
permutation  r1  w3  r2  p3  w2  c3  p1  c1  p2  c2  check
permutation  r1  w3  r2  p3  w2  c3  p2  p1  c1  c2  check
permutation  r1  w3  r2  p3  w2  c3  p2  p1  c2  c1  check
permutation  r1  w3  r2  p3  w2  c3  p2  c2  p1  c1  check
permutation  r1  w3  r2  p3  p1  w2  p2  c3  c1  c2  check
permutation  r1  w3  r2  p3  p1  w2  p2  c3  c2  c1  check
permutation  r1  w3  r2  p3  p1  w2  c3  p2  c1  c2  check
permutation  r1  w3  r2  p3  p1  w2  c3  p2  c2  c1  check
permutation  r1  w3  r2  p3  p1  w2  c3  c1  p2  c2  check
permutation  r1  w3  r2  p3  p1  c3  w2  p2  c1  c2  check
permutation  r1  w3  r2  p3  p1  c3  w2  p2  c2  c1  check
permutation  r1  w3  r2  p3  p1  c3  w2  c1  p2  c2  check
permutation  r1  w3  r2  p3  p1  c3  c1  w2  p2  c2  check
permutation  r1  w3  r2  p3  c3  w2  p1  p2  c1  c2  check
permutation  r1  w3  r2  p3  c3  w2  p1  p2  c2  c1  check
permutation  r1  w3  r2  p3  c3  w2  p1  c1  p2  c2  check
permutation  r1  w3  r2  p3  c3  w2  p2  p1  c1  c2  check
permutation  r1  w3  r2  p3  c3  w2  p2  p1  c2  c1  check
permutation  r1  w3  r2  p3  c3  w2  p2  c2  p1  c1  check
permutation  r1  w3  r2  p3  c3  p1  w2  p2  c1  c2  check
permutation  r1  w3  r2  p3  c3  p1  w2  p2  c2  c1  check
permutation  r1  w3  r2  p3  c3  p1  w2  c1  p2  c2  check
permutation  r1  w3  r2  p3  c3  p1  c1  w2  p2  c2  check
permutation  r1  w3  w2  r2  p1  p2  p3  c3  c1  c2  check
permutation  r1  w3  w2  r2  p1  p2  p3  c3  c2  c1  check
permutation  r1  w3  w2  r2  p1  p3  p2  c3  c1  c2  check
permutation  r1  w3  w2  r2  p1  p3  p2  c3  c2  c1  check
permutation  r1  w3  w2  r2  p1  p3  c3  p2  c1  c2  check
permutation  r1  w3  w2  r2  p1  p3  c3  p2  c2  c1  check
permutation  r1  w3  w2  r2  p1  p3  c3  c1  p2  c2  check
permutation  r1  w3  w2  r2  p2  p1  p3  c3  c1  c2  check
permutation  r1  w3  w2  r2  p2  p1  p3  c3  c2  c1  check
permutation  r1  w3  w2  r2  p2  p3  p1  c3  c1  c2  check
permutation  r1  w3  w2  r2  p2  p3  p1  c3  c2  c1  check
permutation  r1  w3  w2  r2  p2  p3  c3  p1  c1  c2  check
permutation  r1  w3  w2  r2  p2  p3  c3  p1  c2  c1  check
permutation  r1  w3  w2  r2  p2  p3  c3  c2  p1  c1  check
permutation  r1  w3  w2  r2  p3  p1  p2  c3  c1  c2  check
permutation  r1  w3  w2  r2  p3  p1  p2  c3  c2  c1  check
permutation  r1  w3  w2  r2  p3  p1  c3  p2  c1  c2  check
permutation  r1  w3  w2  r2  p3  p1  c3  p2  c2  c1  check
permutation  r1  w3  w2  r2  p3  p1  c3  c1  p2  c2  check
permutation  r1  w3  w2  r2  p3  p2  p1  c3  c1  c2  check
permutation  r1  w3  w2  r2  p3  p2  p1  c3  c2  c1  check
permutation  r1  w3  w2  r2  p3  p2  c3  p1  c1  c2  check
permutation  r1  w3  w2  r2  p3  p2  c3  p1  c2  c1  check
permutation  r1  w3  w2  r2  p3  p2  c3  c2  p1  c1  check
permutation  r1  w3  w2  r2  p3  c3  p1  p2  c1  c2  check
permutation  r1  w3  w2  r2  p3  c3  p1  p2  c2  c1  check
permutation  r1  w3  w2  r2  p3  c3  p1  c1  p2  c2  check
permutation  r1  w3  w2  r2  p3  c3  p2  p1  c1  c2  check
permutation  r1  w3  w2  r2  p3  c3  p2  p1  c2  c1  check
permutation  r1  w3  w2  r2  p3  c3  p2  c2  p1  c1  check
permutation  r1  w3  w2  p1  r2  p2  p3  c3  c1  c2  check
permutation  r1  w3  w2  p1  r2  p2  p3  c3  c2  c1  check
permutation  r1  w3  w2  p1  r2  p3  p2  c3  c1  c2  check
permutation  r1  w3  w2  p1  r2  p3  p2  c3  c2  c1  check
permutation  r1  w3  w2  p1  r2  p3  c3  p2  c1  c2  check
permutation  r1  w3  w2  p1  r2  p3  c3  p2  c2  c1  check
permutation  r1  w3  w2  p1  r2  p3  c3  c1  p2  c2  check
permutation  r1  w3  w2  p1  p3  r2  p2  c3  c1  c2  check
permutation  r1  w3  w2  p1  p3  r2  p2  c3  c2  c1  check
permutation  r1  w3  w2  p1  p3  r2  c3  p2  c1  c2  check
permutation  r1  w3  w2  p1  p3  r2  c3  p2  c2  c1  check
permutation  r1  w3  w2  p1  p3  r2  c3  c1  p2  c2  check
permutation  r1  w3  w2  p1  p3  c3  r2  p2  c1  c2  check
permutation  r1  w3  w2  p1  p3  c3  r2  p2  c2  c1  check
permutation  r1  w3  w2  p1  p3  c3  r2  c1  p2  c2  check
permutation  r1  w3  w2  p1  p3  c3  c1  r2  p2  c2  check
permutation  r1  w3  w2  p3  r2  p1  p2  c3  c1  c2  check
permutation  r1  w3  w2  p3  r2  p1  p2  c3  c2  c1  check
permutation  r1  w3  w2  p3  r2  p1  c3  p2  c1  c2  check
permutation  r1  w3  w2  p3  r2  p1  c3  p2  c2  c1  check
permutation  r1  w3  w2  p3  r2  p1  c3  c1  p2  c2  check
permutation  r1  w3  w2  p3  r2  p2  p1  c3  c1  c2  check
permutation  r1  w3  w2  p3  r2  p2  p1  c3  c2  c1  check
permutation  r1  w3  w2  p3  r2  p2  c3  p1  c1  c2  check
permutation  r1  w3  w2  p3  r2  p2  c3  p1  c2  c1  check
permutation  r1  w3  w2  p3  r2  p2  c3  c2  p1  c1  check
permutation  r1  w3  w2  p3  r2  c3  p1  p2  c1  c2  check
permutation  r1  w3  w2  p3  r2  c3  p1  p2  c2  c1  check
permutation  r1  w3  w2  p3  r2  c3  p1  c1  p2  c2  check
permutation  r1  w3  w2  p3  r2  c3  p2  p1  c1  c2  check
permutation  r1  w3  w2  p3  r2  c3  p2  p1  c2  c1  check
permutation  r1  w3  w2  p3  r2  c3  p2  c2  p1  c1  check
permutation  r1  w3  w2  p3  p1  r2  p2  c3  c1  c2  check
permutation  r1  w3  w2  p3  p1  r2  p2  c3  c2  c1  check
permutation  r1  w3  w2  p3  p1  r2  c3  p2  c1  c2  check
permutation  r1  w3  w2  p3  p1  r2  c3  p2  c2  c1  check
permutation  r1  w3  w2  p3  p1  r2  c3  c1  p2  c2  check
permutation  r1  w3  w2  p3  p1  c3  r2  p2  c1  c2  check
permutation  r1  w3  w2  p3  p1  c3  r2  p2  c2  c1  check
permutation  r1  w3  w2  p3  p1  c3  r2  c1  p2  c2  check
permutation  r1  w3  w2  p3  p1  c3  c1  r2  p2  c2  check
permutation  r1  w3  w2  p3  c3  r2  p1  p2  c1  c2  check
permutation  r1  w3  w2  p3  c3  r2  p1  p2  c2  c1  check
permutation  r1  w3  w2  p3  c3  r2  p1  c1  p2  c2  check
permutation  r1  w3  w2  p3  c3  r2  p2  p1  c1  c2  check
permutation  r1  w3  w2  p3  c3  r2  p2  p1  c2  c1  check
permutation  r1  w3  w2  p3  c3  r2  p2  c2  p1  c1  check
permutation  r1  w3  w2  p3  c3  p1  r2  p2  c1  c2  check
permutation  r1  w3  w2  p3  c3  p1  r2  p2  c2  c1  check
permutation  r1  w3  w2  p3  c3  p1  r2  c1  p2  c2  check
permutation  r1  w3  w2  p3  c3  p1  c1  r2  p2  c2  check
permutation  r1  w3  p1  r2  w2  p2  p3  c3  c1  c2  check
permutation  r1  w3  p1  r2  w2  p2  p3  c3  c2  c1  check
permutation  r1  w3  p1  r2  w2  p3  p2  c3  c1  c2  check
permutation  r1  w3  p1  r2  w2  p3  p2  c3  c2  c1  check
permutation  r1  w3  p1  r2  w2  p3  c3  p2  c1  c2  check
permutation  r1  w3  p1  r2  w2  p3  c3  p2  c2  c1  check
permutation  r1  w3  p1  r2  w2  p3  c3  c1  p2  c2  check
permutation  r1  w3  p1  r2  p3  w2  p2  c3  c1  c2  check
permutation  r1  w3  p1  r2  p3  w2  p2  c3  c2  c1  check
permutation  r1  w3  p1  r2  p3  w2  c3  p2  c1  c2  check
permutation  r1  w3  p1  r2  p3  w2  c3  p2  c2  c1  check
permutation  r1  w3  p1  r2  p3  w2  c3  c1  p2  c2  check
permutation  r1  w3  p1  r2  p3  c3  w2  p2  c1  c2  check
permutation  r1  w3  p1  r2  p3  c3  w2  p2  c2  c1  check
permutation  r1  w3  p1  r2  p3  c3  w2  c1  p2  c2  check
permutation  r1  w3  p1  r2  p3  c3  c1  w2  p2  c2  check
permutation  r1  w3  p1  w2  r2  p2  p3  c3  c1  c2  check
permutation  r1  w3  p1  w2  r2  p2  p3  c3  c2  c1  check
permutation  r1  w3  p1  w2  r2  p3  p2  c3  c1  c2  check
permutation  r1  w3  p1  w2  r2  p3  p2  c3  c2  c1  check
permutation  r1  w3  p1  w2  r2  p3  c3  p2  c1  c2  check
permutation  r1  w3  p1  w2  r2  p3  c3  p2  c2  c1  check
permutation  r1  w3  p1  w2  r2  p3  c3  c1  p2  c2  check
permutation  r1  w3  p1  w2  p3  r2  p2  c3  c1  c2  check
permutation  r1  w3  p1  w2  p3  r2  p2  c3  c2  c1  check
permutation  r1  w3  p1  w2  p3  r2  c3  p2  c1  c2  check
permutation  r1  w3  p1  w2  p3  r2  c3  p2  c2  c1  check
permutation  r1  w3  p1  w2  p3  r2  c3  c1  p2  c2  check
permutation  r1  w3  p1  w2  p3  c3  r2  p2  c1  c2  check
permutation  r1  w3  p1  w2  p3  c3  r2  p2  c2  c1  check
permutation  r1  w3  p1  w2  p3  c3  r2  c1  p2  c2  check
permutation  r1  w3  p1  w2  p3  c3  c1  r2  p2  c2  check
permutation  r1  w3  p1  p3  r2  w2  p2  c3  c1  c2  check
permutation  r1  w3  p1  p3  r2  w2  p2  c3  c2  c1  check
permutation  r1  w3  p1  p3  r2  w2  c3  p2  c1  c2  check
permutation  r1  w3  p1  p3  r2  w2  c3  p2  c2  c1  check
permutation  r1  w3  p1  p3  r2  w2  c3  c1  p2  c2  check
permutation  r1  w3  p1  p3  r2  c3  w2  p2  c1  c2  check
permutation  r1  w3  p1  p3  r2  c3  w2  p2  c2  c1  check
permutation  r1  w3  p1  p3  r2  c3  w2  c1  p2  c2  check
permutation  r1  w3  p1  p3  r2  c3  c1  w2  p2  c2  check
permutation  r1  w3  p1  p3  w2  r2  p2  c3  c1  c2  check
permutation  r1  w3  p1  p3  w2  r2  p2  c3  c2  c1  check
permutation  r1  w3  p1  p3  w2  r2  c3  p2  c1  c2  check
permutation  r1  w3  p1  p3  w2  r2  c3  p2  c2  c1  check
permutation  r1  w3  p1  p3  w2  r2  c3  c1  p2  c2  check
permutation  r1  w3  p1  p3  w2  c3  r2  p2  c1  c2  check
permutation  r1  w3  p1  p3  w2  c3  r2  p2  c2  c1  check
permutation  r1  w3  p1  p3  w2  c3  r2  c1  p2  c2  check
permutation  r1  w3  p1  p3  w2  c3  c1  r2  p2  c2  check
permutation  r1  w3  p1  p3  c3  r2  w2  p2  c1  c2  check
permutation  r1  w3  p1  p3  c3  r2  w2  p2  c2  c1  check
permutation  r1  w3  p1  p3  c3  r2  w2  c1  p2  c2  check
permutation  r1  w3  p1  p3  c3  r2  c1  w2  p2  c2  check
permutation  r1  w3  p1  p3  c3  w2  r2  p2  c1  c2  check
permutation  r1  w3  p1  p3  c3  w2  r2  p2  c2  c1  check
permutation  r1  w3  p1  p3  c3  w2  r2  c1  p2  c2  check
permutation  r1  w3  p1  p3  c3  w2  c1  r2  p2  c2  check
permutation  r1  w3  p1  p3  c3  c1  r2  w2  p2  c2  check
permutation  r1  w3  p1  p3  c3  c1  w2  r2  p2  c2  check
permutation  r1  w3  p3  r2  w2  p1  p2  c3  c1  c2  check
permutation  r1  w3  p3  r2  w2  p1  p2  c3  c2  c1  check
permutation  r1  w3  p3  r2  w2  p1  c3  p2  c1  c2  check
permutation  r1  w3  p3  r2  w2  p1  c3  p2  c2  c1  check
permutation  r1  w3  p3  r2  w2  p1  c3  c1  p2  c2  check
permutation  r1  w3  p3  r2  w2  p2  p1  c3  c1  c2  check
permutation  r1  w3  p3  r2  w2  p2  p1  c3  c2  c1  check
permutation  r1  w3  p3  r2  w2  p2  c3  p1  c1  c2  check
permutation  r1  w3  p3  r2  w2  p2  c3  p1  c2  c1  check
permutation  r1  w3  p3  r2  w2  p2  c3  c2  p1  c1  check
permutation  r1  w3  p3  r2  w2  c3  p1  p2  c1  c2  check
permutation  r1  w3  p3  r2  w2  c3  p1  p2  c2  c1  check
permutation  r1  w3  p3  r2  w2  c3  p1  c1  p2  c2  check
permutation  r1  w3  p3  r2  w2  c3  p2  p1  c1  c2  check
permutation  r1  w3  p3  r2  w2  c3  p2  p1  c2  c1  check
permutation  r1  w3  p3  r2  w2  c3  p2  c2  p1  c1  check
permutation  r1  w3  p3  r2  p1  w2  p2  c3  c1  c2  check
permutation  r1  w3  p3  r2  p1  w2  p2  c3  c2  c1  check
permutation  r1  w3  p3  r2  p1  w2  c3  p2  c1  c2  check
permutation  r1  w3  p3  r2  p1  w2  c3  p2  c2  c1  check
permutation  r1  w3  p3  r2  p1  w2  c3  c1  p2  c2  check
permutation  r1  w3  p3  r2  p1  c3  w2  p2  c1  c2  check
permutation  r1  w3  p3  r2  p1  c3  w2  p2  c2  c1  check
permutation  r1  w3  p3  r2  p1  c3  w2  c1  p2  c2  check
permutation  r1  w3  p3  r2  p1  c3  c1  w2  p2  c2  check
permutation  r1  w3  p3  r2  c3  w2  p1  p2  c1  c2  check
permutation  r1  w3  p3  r2  c3  w2  p1  p2  c2  c1  check
permutation  r1  w3  p3  r2  c3  w2  p1  c1  p2  c2  check
permutation  r1  w3  p3  r2  c3  w2  p2  p1  c1  c2  check
permutation  r1  w3  p3  r2  c3  w2  p2  p1  c2  c1  check
permutation  r1  w3  p3  r2  c3  w2  p2  c2  p1  c1  check
permutation  r1  w3  p3  r2  c3  p1  w2  p2  c1  c2  check
permutation  r1  w3  p3  r2  c3  p1  w2  p2  c2  c1  check
permutation  r1  w3  p3  r2  c3  p1  w2  c1  p2  c2  check
permutation  r1  w3  p3  r2  c3  p1  c1  w2  p2  c2  check
permutation  r1  w3  p3  w2  r2  p1  p2  c3  c1  c2  check
permutation  r1  w3  p3  w2  r2  p1  p2  c3  c2  c1  check
permutation  r1  w3  p3  w2  r2  p1  c3  p2  c1  c2  check
permutation  r1  w3  p3  w2  r2  p1  c3  p2  c2  c1  check
permutation  r1  w3  p3  w2  r2  p1  c3  c1  p2  c2  check
permutation  r1  w3  p3  w2  r2  p2  p1  c3  c1  c2  check
permutation  r1  w3  p3  w2  r2  p2  p1  c3  c2  c1  check
permutation  r1  w3  p3  w2  r2  p2  c3  p1  c1  c2  check
permutation  r1  w3  p3  w2  r2  p2  c3  p1  c2  c1  check
permutation  r1  w3  p3  w2  r2  p2  c3  c2  p1  c1  check
permutation  r1  w3  p3  w2  r2  c3  p1  p2  c1  c2  check
permutation  r1  w3  p3  w2  r2  c3  p1  p2  c2  c1  check
permutation  r1  w3  p3  w2  r2  c3  p1  c1  p2  c2  check
permutation  r1  w3  p3  w2  r2  c3  p2  p1  c1  c2  check
permutation  r1  w3  p3  w2  r2  c3  p2  p1  c2  c1  check
permutation  r1  w3  p3  w2  r2  c3  p2  c2  p1  c1  check
permutation  r1  w3  p3  w2  p1  r2  p2  c3  c1  c2  check
permutation  r1  w3  p3  w2  p1  r2  p2  c3  c2  c1  check
permutation  r1  w3  p3  w2  p1  r2  c3  p2  c1  c2  check
permutation  r1  w3  p3  w2  p1  r2  c3  p2  c2  c1  check
permutation  r1  w3  p3  w2  p1  r2  c3  c1  p2  c2  check
permutation  r1  w3  p3  w2  p1  c3  r2  p2  c1  c2  check
permutation  r1  w3  p3  w2  p1  c3  r2  p2  c2  c1  check
permutation  r1  w3  p3  w2  p1  c3  r2  c1  p2  c2  check
permutation  r1  w3  p3  w2  p1  c3  c1  r2  p2  c2  check
permutation  r1  w3  p3  w2  c3  r2  p1  p2  c1  c2  check
permutation  r1  w3  p3  w2  c3  r2  p1  p2  c2  c1  check
permutation  r1  w3  p3  w2  c3  r2  p1  c1  p2  c2  check
permutation  r1  w3  p3  w2  c3  r2  p2  p1  c1  c2  check
permutation  r1  w3  p3  w2  c3  r2  p2  p1  c2  c1  check
permutation  r1  w3  p3  w2  c3  r2  p2  c2  p1  c1  check
permutation  r1  w3  p3  w2  c3  p1  r2  p2  c1  c2  check
permutation  r1  w3  p3  w2  c3  p1  r2  p2  c2  c1  check
permutation  r1  w3  p3  w2  c3  p1  r2  c1  p2  c2  check
permutation  r1  w3  p3  w2  c3  p1  c1  r2  p2  c2  check
permutation  r1  w3  p3  p1  r2  w2  p2  c3  c1  c2  check
permutation  r1  w3  p3  p1  r2  w2  p2  c3  c2  c1  check
permutation  r1  w3  p3  p1  r2  w2  c3  p2  c1  c2  check
permutation  r1  w3  p3  p1  r2  w2  c3  p2  c2  c1  check
permutation  r1  w3  p3  p1  r2  w2  c3  c1  p2  c2  check
permutation  r1  w3  p3  p1  r2  c3  w2  p2  c1  c2  check
permutation  r1  w3  p3  p1  r2  c3  w2  p2  c2  c1  check
permutation  r1  w3  p3  p1  r2  c3  w2  c1  p2  c2  check
permutation  r1  w3  p3  p1  r2  c3  c1  w2  p2  c2  check
permutation  r1  w3  p3  p1  w2  r2  p2  c3  c1  c2  check
permutation  r1  w3  p3  p1  w2  r2  p2  c3  c2  c1  check
permutation  r1  w3  p3  p1  w2  r2  c3  p2  c1  c2  check
permutation  r1  w3  p3  p1  w2  r2  c3  p2  c2  c1  check
permutation  r1  w3  p3  p1  w2  r2  c3  c1  p2  c2  check
permutation  r1  w3  p3  p1  w2  c3  r2  p2  c1  c2  check
permutation  r1  w3  p3  p1  w2  c3  r2  p2  c2  c1  check
permutation  r1  w3  p3  p1  w2  c3  r2  c1  p2  c2  check
permutation  r1  w3  p3  p1  w2  c3  c1  r2  p2  c2  check
permutation  r1  w3  p3  p1  c3  r2  w2  p2  c1  c2  check
permutation  r1  w3  p3  p1  c3  r2  w2  p2  c2  c1  check
permutation  r1  w3  p3  p1  c3  r2  w2  c1  p2  c2  check
permutation  r1  w3  p3  p1  c3  r2  c1  w2  p2  c2  check
permutation  r1  w3  p3  p1  c3  w2  r2  p2  c1  c2  check
permutation  r1  w3  p3  p1  c3  w2  r2  p2  c2  c1  check
permutation  r1  w3  p3  p1  c3  w2  r2  c1  p2  c2  check
permutation  r1  w3  p3  p1  c3  w2  c1  r2  p2  c2  check
permutation  r1  w3  p3  p1  c3  c1  r2  w2  p2  c2  check
permutation  r1  w3  p3  p1  c3  c1  w2  r2  p2  c2  check
permutation  r1  w3  p3  c3  r2  w2  p1  p2  c1  c2  check
permutation  r1  w3  p3  c3  r2  w2  p1  p2  c2  c1  check
permutation  r1  w3  p3  c3  r2  w2  p1  c1  p2  c2  check
permutation  r1  w3  p3  c3  r2  w2  p2  p1  c1  c2  check
permutation  r1  w3  p3  c3  r2  w2  p2  p1  c2  c1  check
permutation  r1  w3  p3  c3  r2  w2  p2  c2  p1  c1  check
permutation  r1  w3  p3  c3  r2  p1  w2  p2  c1  c2  check
permutation  r1  w3  p3  c3  r2  p1  w2  p2  c2  c1  check
permutation  r1  w3  p3  c3  r2  p1  w2  c1  p2  c2  check
permutation  r1  w3  p3  c3  r2  p1  c1  w2  p2  c2  check
permutation  r1  w3  p3  c3  w2  r2  p1  p2  c1  c2  check
permutation  r1  w3  p3  c3  w2  r2  p1  p2  c2  c1  check
permutation  r1  w3  p3  c3  w2  r2  p1  c1  p2  c2  check
permutation  r1  w3  p3  c3  w2  r2  p2  p1  c1  c2  check
permutation  r1  w3  p3  c3  w2  r2  p2  p1  c2  c1  check
permutation  r1  w3  p3  c3  w2  r2  p2  c2  p1  c1  check
permutation  r1  w3  p3  c3  w2  p1  r2  p2  c1  c2  check
permutation  r1  w3  p3  c3  w2  p1  r2  p2  c2  c1  check
permutation  r1  w3  p3  c3  w2  p1  r2  c1  p2  c2  check
permutation  r1  w3  p3  c3  w2  p1  c1  r2  p2  c2  check
permutation  r1  w3  p3  c3  p1  r2  w2  p2  c1  c2  check
permutation  r1  w3  p3  c3  p1  r2  w2  p2  c2  c1  check
permutation  r1  w3  p3  c3  p1  r2  w2  c1  p2  c2  check
permutation  r1  w3  p3  c3  p1  r2  c1  w2  p2  c2  check
permutation  r1  w3  p3  c3  p1  w2  r2  p2  c1  c2  check
permutation  r1  w3  p3  c3  p1  w2  r2  p2  c2  c1  check
permutation  r1  w3  p3  c3  p1  w2  r2  c1  p2  c2  check
permutation  r1  w3  p3  c3  p1  w2  c1  r2  p2  c2  check
permutation  r1  w3  p3  c3  p1  c1  r2  w2  p2  c2  check
permutation  r1  w3  p3  c3  p1  c1  w2  r2  p2  c2  check
permutation  r1  p1  r2  w2  w3  p2  p3  c3  c1  c2  check
permutation  r1  p1  r2  w2  w3  p2  p3  c3  c2  c1  check
permutation  r1  p1  r2  w2  w3  p3  p2  c3  c1  c2  check
permutation  r1  p1  r2  w2  w3  p3  p2  c3  c2  c1  check
permutation  r1  p1  r2  w2  w3  p3  c3  p2  c1  c2  check
permutation  r1  p1  r2  w2  w3  p3  c3  p2  c2  c1  check
permutation  r1  p1  r2  w2  w3  p3  c3  c1  p2  c2  check
permutation  r1  p1  r2  w2  p2  w3  p3  c3  c1  c2  check
permutation  r1  p1  r2  w2  p2  w3  p3  c3  c2  c1  check
permutation  r1  p1  w2  w3  r2  p2  p3  c3  c1  c2  check
permutation  r1  p1  w2  w3  r2  p2  p3  c3  c2  c1  check
permutation  r1  p1  w2  w3  r2  p3  p2  c3  c1  c2  check
permutation  r1  p1  w2  w3  r2  p3  p2  c3  c2  c1  check
permutation  r1  p1  w2  w3  r2  p3  c3  p2  c1  c2  check
permutation  r1  p1  w2  w3  r2  p3  c3  p2  c2  c1  check
permutation  r1  p1  w2  w3  r2  p3  c3  c1  p2  c2  check
permutation  r1  p1  w2  w3  p3  r2  p2  c3  c1  c2  check
permutation  r1  p1  w2  w3  p3  r2  p2  c3  c2  c1  check
permutation  r1  p1  w2  w3  p3  r2  c3  p2  c1  c2  check
permutation  r1  p1  w2  w3  p3  r2  c3  p2  c2  c1  check
permutation  r1  p1  w2  w3  p3  r2  c3  c1  p2  c2  check
permutation  r1  p1  w2  w3  p3  c3  r2  p2  c1  c2  check
permutation  r1  p1  w2  w3  p3  c3  r2  p2  c2  c1  check
permutation  r1  p1  w2  w3  p3  c3  r2  c1  p2  c2  check
permutation  r1  p1  w2  w3  p3  c3  c1  r2  p2  c2  check
permutation  r1  p1  w3  r2  w2  p2  p3  c3  c1  c2  check
permutation  r1  p1  w3  r2  w2  p2  p3  c3  c2  c1  check
permutation  r1  p1  w3  r2  w2  p3  p2  c3  c1  c2  check
permutation  r1  p1  w3  r2  w2  p3  p2  c3  c2  c1  check
permutation  r1  p1  w3  r2  w2  p3  c3  p2  c1  c2  check
permutation  r1  p1  w3  r2  w2  p3  c3  p2  c2  c1  check
permutation  r1  p1  w3  r2  w2  p3  c3  c1  p2  c2  check
permutation  r1  p1  w3  r2  p3  w2  p2  c3  c1  c2  check
permutation  r1  p1  w3  r2  p3  w2  p2  c3  c2  c1  check
permutation  r1  p1  w3  r2  p3  w2  c3  p2  c1  c2  check
permutation  r1  p1  w3  r2  p3  w2  c3  p2  c2  c1  check
permutation  r1  p1  w3  r2  p3  w2  c3  c1  p2  c2  check
permutation  r1  p1  w3  r2  p3  c3  w2  p2  c1  c2  check
permutation  r1  p1  w3  r2  p3  c3  w2  p2  c2  c1  check
permutation  r1  p1  w3  r2  p3  c3  w2  c1  p2  c2  check
permutation  r1  p1  w3  r2  p3  c3  c1  w2  p2  c2  check
permutation  r1  p1  w3  w2  r2  p2  p3  c3  c1  c2  check
permutation  r1  p1  w3  w2  r2  p2  p3  c3  c2  c1  check
permutation  r1  p1  w3  w2  r2  p3  p2  c3  c1  c2  check
permutation  r1  p1  w3  w2  r2  p3  p2  c3  c2  c1  check
permutation  r1  p1  w3  w2  r2  p3  c3  p2  c1  c2  check
permutation  r1  p1  w3  w2  r2  p3  c3  p2  c2  c1  check
permutation  r1  p1  w3  w2  r2  p3  c3  c1  p2  c2  check
permutation  r1  p1  w3  w2  p3  r2  p2  c3  c1  c2  check
permutation  r1  p1  w3  w2  p3  r2  p2  c3  c2  c1  check
permutation  r1  p1  w3  w2  p3  r2  c3  p2  c1  c2  check
permutation  r1  p1  w3  w2  p3  r2  c3  p2  c2  c1  check
permutation  r1  p1  w3  w2  p3  r2  c3  c1  p2  c2  check
permutation  r1  p1  w3  w2  p3  c3  r2  p2  c1  c2  check
permutation  r1  p1  w3  w2  p3  c3  r2  p2  c2  c1  check
permutation  r1  p1  w3  w2  p3  c3  r2  c1  p2  c2  check
permutation  r1  p1  w3  w2  p3  c3  c1  r2  p2  c2  check
permutation  r1  p1  w3  p3  r2  w2  p2  c3  c1  c2  check
permutation  r1  p1  w3  p3  r2  w2  p2  c3  c2  c1  check
permutation  r1  p1  w3  p3  r2  w2  c3  p2  c1  c2  check
permutation  r1  p1  w3  p3  r2  w2  c3  p2  c2  c1  check
permutation  r1  p1  w3  p3  r2  w2  c3  c1  p2  c2  check
permutation  r1  p1  w3  p3  r2  c3  w2  p2  c1  c2  check
permutation  r1  p1  w3  p3  r2  c3  w2  p2  c2  c1  check
permutation  r1  p1  w3  p3  r2  c3  w2  c1  p2  c2  check
permutation  r1  p1  w3  p3  r2  c3  c1  w2  p2  c2  check
permutation  r1  p1  w3  p3  w2  r2  p2  c3  c1  c2  check
permutation  r1  p1  w3  p3  w2  r2  p2  c3  c2  c1  check
permutation  r1  p1  w3  p3  w2  r2  c3  p2  c1  c2  check
permutation  r1  p1  w3  p3  w2  r2  c3  p2  c2  c1  check
permutation  r1  p1  w3  p3  w2  r2  c3  c1  p2  c2  check
permutation  r1  p1  w3  p3  w2  c3  r2  p2  c1  c2  check
permutation  r1  p1  w3  p3  w2  c3  r2  p2  c2  c1  check
permutation  r1  p1  w3  p3  w2  c3  r2  c1  p2  c2  check
permutation  r1  p1  w3  p3  w2  c3  c1  r2  p2  c2  check
permutation  r1  p1  w3  p3  c3  r2  w2  p2  c1  c2  check
permutation  r1  p1  w3  p3  c3  r2  w2  p2  c2  c1  check
permutation  r1  p1  w3  p3  c3  r2  w2  c1  p2  c2  check
permutation  r1  p1  w3  p3  c3  r2  c1  w2  p2  c2  check
permutation  r1  p1  w3  p3  c3  w2  r2  p2  c1  c2  check
permutation  r1  p1  w3  p3  c3  w2  r2  p2  c2  c1  check
permutation  r1  p1  w3  p3  c3  w2  r2  c1  p2  c2  check
permutation  r1  p1  w3  p3  c3  w2  c1  r2  p2  c2  check
permutation  r1  p1  w3  p3  c3  c1  r2  w2  p2  c2  check
permutation  r1  p1  w3  p3  c3  c1  w2  r2  p2  c2  check
permutation  w2  r1  r2  w3  p1  p2  p3  c3  c1  c2  check
permutation  w2  r1  r2  w3  p1  p2  p3  c3  c2  c1  check
permutation  w2  r1  r2  w3  p1  p3  p2  c3  c1  c2  check
permutation  w2  r1  r2  w3  p1  p3  p2  c3  c2  c1  check
permutation  w2  r1  r2  w3  p1  p3  c3  p2  c1  c2  check
permutation  w2  r1  r2  w3  p1  p3  c3  p2  c2  c1  check
permutation  w2  r1  r2  w3  p1  p3  c3  c1  p2  c2  check
permutation  w2  r1  r2  w3  p2  p1  p3  c3  c1  c2  check
permutation  w2  r1  r2  w3  p2  p1  p3  c3  c2  c1  check
permutation  w2  r1  r2  w3  p2  p3  p1  c3  c1  c2  check
permutation  w2  r1  r2  w3  p2  p3  p1  c3  c2  c1  check
permutation  w2  r1  r2  w3  p2  p3  c3  p1  c1  c2  check
permutation  w2  r1  r2  w3  p2  p3  c3  p1  c2  c1  check
permutation  w2  r1  r2  w3  p2  p3  c3  c2  p1  c1  check
permutation  w2  r1  r2  w3  p3  p1  p2  c3  c1  c2  check
permutation  w2  r1  r2  w3  p3  p1  p2  c3  c2  c1  check
permutation  w2  r1  r2  w3  p3  p1  c3  p2  c1  c2  check
permutation  w2  r1  r2  w3  p3  p1  c3  p2  c2  c1  check
permutation  w2  r1  r2  w3  p3  p1  c3  c1  p2  c2  check
permutation  w2  r1  r2  w3  p3  p2  p1  c3  c1  c2  check
permutation  w2  r1  r2  w3  p3  p2  p1  c3  c2  c1  check
permutation  w2  r1  r2  w3  p3  p2  c3  p1  c1  c2  check
permutation  w2  r1  r2  w3  p3  p2  c3  p1  c2  c1  check
permutation  w2  r1  r2  w3  p3  p2  c3  c2  p1  c1  check
permutation  w2  r1  r2  w3  p3  c3  p1  p2  c1  c2  check
permutation  w2  r1  r2  w3  p3  c3  p1  p2  c2  c1  check
permutation  w2  r1  r2  w3  p3  c3  p1  c1  p2  c2  check
permutation  w2  r1  r2  w3  p3  c3  p2  p1  c1  c2  check
permutation  w2  r1  r2  w3  p3  c3  p2  p1  c2  c1  check
permutation  w2  r1  r2  w3  p3  c3  p2  c2  p1  c1  check
permutation  w2  r1  r2  p1  w3  p2  p3  c3  c1  c2  check
permutation  w2  r1  r2  p1  w3  p2  p3  c3  c2  c1  check
permutation  w2  r1  r2  p1  w3  p3  p2  c3  c1  c2  check
permutation  w2  r1  r2  p1  w3  p3  p2  c3  c2  c1  check
permutation  w2  r1  r2  p1  w3  p3  c3  p2  c1  c2  check
permutation  w2  r1  r2  p1  w3  p3  c3  p2  c2  c1  check
permutation  w2  r1  r2  p1  w3  p3  c3  c1  p2  c2  check
permutation  w2  r1  r2  p1  p2  w3  p3  c3  c1  c2  check
permutation  w2  r1  r2  p1  p2  w3  p3  c3  c2  c1  check
permutation  w2  r1  r2  p2  w3  p1  p3  c3  c1  c2  check
permutation  w2  r1  r2  p2  w3  p1  p3  c3  c2  c1  check
permutation  w2  r1  r2  p2  w3  p3  p1  c3  c1  c2  check
permutation  w2  r1  r2  p2  w3  p3  p1  c3  c2  c1  check
permutation  w2  r1  r2  p2  w3  p3  c3  p1  c1  c2  check
permutation  w2  r1  r2  p2  w3  p3  c3  p1  c2  c1  check
permutation  w2  r1  r2  p2  w3  p3  c3  c2  p1  c1  check
permutation  w2  r1  r2  p2  p1  w3  p3  c3  c1  c2  check
permutation  w2  r1  r2  p2  p1  w3  p3  c3  c2  c1  check
permutation  w2  r1  w3  r2  p1  p2  p3  c3  c1  c2  check
permutation  w2  r1  w3  r2  p1  p2  p3  c3  c2  c1  check
permutation  w2  r1  w3  r2  p1  p3  p2  c3  c1  c2  check
permutation  w2  r1  w3  r2  p1  p3  p2  c3  c2  c1  check
permutation  w2  r1  w3  r2  p1  p3  c3  p2  c1  c2  check
permutation  w2  r1  w3  r2  p1  p3  c3  p2  c2  c1  check
permutation  w2  r1  w3  r2  p1  p3  c3  c1  p2  c2  check
permutation  w2  r1  w3  r2  p2  p1  p3  c3  c1  c2  check
permutation  w2  r1  w3  r2  p2  p1  p3  c3  c2  c1  check
permutation  w2  r1  w3  r2  p2  p3  p1  c3  c1  c2  check
permutation  w2  r1  w3  r2  p2  p3  p1  c3  c2  c1  check
permutation  w2  r1  w3  r2  p2  p3  c3  p1  c1  c2  check
permutation  w2  r1  w3  r2  p2  p3  c3  p1  c2  c1  check
permutation  w2  r1  w3  r2  p2  p3  c3  c2  p1  c1  check
permutation  w2  r1  w3  r2  p3  p1  p2  c3  c1  c2  check
permutation  w2  r1  w3  r2  p3  p1  p2  c3  c2  c1  check
permutation  w2  r1  w3  r2  p3  p1  c3  p2  c1  c2  check
permutation  w2  r1  w3  r2  p3  p1  c3  p2  c2  c1  check
permutation  w2  r1  w3  r2  p3  p1  c3  c1  p2  c2  check
permutation  w2  r1  w3  r2  p3  p2  p1  c3  c1  c2  check
permutation  w2  r1  w3  r2  p3  p2  p1  c3  c2  c1  check
permutation  w2  r1  w3  r2  p3  p2  c3  p1  c1  c2  check
permutation  w2  r1  w3  r2  p3  p2  c3  p1  c2  c1  check
permutation  w2  r1  w3  r2  p3  p2  c3  c2  p1  c1  check
permutation  w2  r1  w3  r2  p3  c3  p1  p2  c1  c2  check
permutation  w2  r1  w3  r2  p3  c3  p1  p2  c2  c1  check
permutation  w2  r1  w3  r2  p3  c3  p1  c1  p2  c2  check
permutation  w2  r1  w3  r2  p3  c3  p2  p1  c1  c2  check
permutation  w2  r1  w3  r2  p3  c3  p2  p1  c2  c1  check
permutation  w2  r1  w3  r2  p3  c3  p2  c2  p1  c1  check
permutation  w2  r1  w3  p1  r2  p2  p3  c3  c1  c2  check
permutation  w2  r1  w3  p1  r2  p2  p3  c3  c2  c1  check
permutation  w2  r1  w3  p1  r2  p3  p2  c3  c1  c2  check
permutation  w2  r1  w3  p1  r2  p3  p2  c3  c2  c1  check
permutation  w2  r1  w3  p1  r2  p3  c3  p2  c1  c2  check
permutation  w2  r1  w3  p1  r2  p3  c3  p2  c2  c1  check
permutation  w2  r1  w3  p1  r2  p3  c3  c1  p2  c2  check
permutation  w2  r1  w3  p1  p3  r2  p2  c3  c1  c2  check
permutation  w2  r1  w3  p1  p3  r2  p2  c3  c2  c1  check
permutation  w2  r1  w3  p1  p3  r2  c3  p2  c1  c2  check
permutation  w2  r1  w3  p1  p3  r2  c3  p2  c2  c1  check
permutation  w2  r1  w3  p1  p3  r2  c3  c1  p2  c2  check
permutation  w2  r1  w3  p1  p3  c3  r2  p2  c1  c2  check
permutation  w2  r1  w3  p1  p3  c3  r2  p2  c2  c1  check
permutation  w2  r1  w3  p1  p3  c3  r2  c1  p2  c2  check
permutation  w2  r1  w3  p1  p3  c3  c1  r2  p2  c2  check
permutation  w2  r1  w3  p3  r2  p1  p2  c3  c1  c2  check
permutation  w2  r1  w3  p3  r2  p1  p2  c3  c2  c1  check
permutation  w2  r1  w3  p3  r2  p1  c3  p2  c1  c2  check
permutation  w2  r1  w3  p3  r2  p1  c3  p2  c2  c1  check
permutation  w2  r1  w3  p3  r2  p1  c3  c1  p2  c2  check
permutation  w2  r1  w3  p3  r2  p2  p1  c3  c1  c2  check
permutation  w2  r1  w3  p3  r2  p2  p1  c3  c2  c1  check
permutation  w2  r1  w3  p3  r2  p2  c3  p1  c1  c2  check
permutation  w2  r1  w3  p3  r2  p2  c3  p1  c2  c1  check
permutation  w2  r1  w3  p3  r2  p2  c3  c2  p1  c1  check
permutation  w2  r1  w3  p3  r2  c3  p1  p2  c1  c2  check
permutation  w2  r1  w3  p3  r2  c3  p1  p2  c2  c1  check
permutation  w2  r1  w3  p3  r2  c3  p1  c1  p2  c2  check
permutation  w2  r1  w3  p3  r2  c3  p2  p1  c1  c2  check
permutation  w2  r1  w3  p3  r2  c3  p2  p1  c2  c1  check
permutation  w2  r1  w3  p3  r2  c3  p2  c2  p1  c1  check
permutation  w2  r1  w3  p3  p1  r2  p2  c3  c1  c2  check
permutation  w2  r1  w3  p3  p1  r2  p2  c3  c2  c1  check
permutation  w2  r1  w3  p3  p1  r2  c3  p2  c1  c2  check
permutation  w2  r1  w3  p3  p1  r2  c3  p2  c2  c1  check
permutation  w2  r1  w3  p3  p1  r2  c3  c1  p2  c2  check
permutation  w2  r1  w3  p3  p1  c3  r2  p2  c1  c2  check
permutation  w2  r1  w3  p3  p1  c3  r2  p2  c2  c1  check
permutation  w2  r1  w3  p3  p1  c3  r2  c1  p2  c2  check
permutation  w2  r1  w3  p3  p1  c3  c1  r2  p2  c2  check
permutation  w2  r1  w3  p3  c3  r2  p1  p2  c1  c2  check
permutation  w2  r1  w3  p3  c3  r2  p1  p2  c2  c1  check
permutation  w2  r1  w3  p3  c3  r2  p1  c1  p2  c2  check
permutation  w2  r1  w3  p3  c3  r2  p2  p1  c1  c2  check
permutation  w2  r1  w3  p3  c3  r2  p2  p1  c2  c1  check
permutation  w2  r1  w3  p3  c3  r2  p2  c2  p1  c1  check
permutation  w2  r1  w3  p3  c3  p1  r2  p2  c1  c2  check
permutation  w2  r1  w3  p3  c3  p1  r2  p2  c2  c1  check
permutation  w2  r1  w3  p3  c3  p1  r2  c1  p2  c2  check
permutation  w2  r1  w3  p3  c3  p1  c1  r2  p2  c2  check
permutation  w2  r1  p1  r2  w3  p2  p3  c3  c1  c2  check
permutation  w2  r1  p1  r2  w3  p2  p3  c3  c2  c1  check
permutation  w2  r1  p1  r2  w3  p3  p2  c3  c1  c2  check
permutation  w2  r1  p1  r2  w3  p3  p2  c3  c2  c1  check
permutation  w2  r1  p1  r2  w3  p3  c3  p2  c1  c2  check
permutation  w2  r1  p1  r2  w3  p3  c3  p2  c2  c1  check
permutation  w2  r1  p1  r2  w3  p3  c3  c1  p2  c2  check
permutation  w2  r1  p1  r2  p2  w3  p3  c3  c1  c2  check
permutation  w2  r1  p1  r2  p2  w3  p3  c3  c2  c1  check
permutation  w2  r1  p1  w3  r2  p2  p3  c3  c1  c2  check
permutation  w2  r1  p1  w3  r2  p2  p3  c3  c2  c1  check
permutation  w2  r1  p1  w3  r2  p3  p2  c3  c1  c2  check
permutation  w2  r1  p1  w3  r2  p3  p2  c3  c2  c1  check
permutation  w2  r1  p1  w3  r2  p3  c3  p2  c1  c2  check
permutation  w2  r1  p1  w3  r2  p3  c3  p2  c2  c1  check
permutation  w2  r1  p1  w3  r2  p3  c3  c1  p2  c2  check
permutation  w2  r1  p1  w3  p3  r2  p2  c3  c1  c2  check
permutation  w2  r1  p1  w3  p3  r2  p2  c3  c2  c1  check
permutation  w2  r1  p1  w3  p3  r2  c3  p2  c1  c2  check
permutation  w2  r1  p1  w3  p3  r2  c3  p2  c2  c1  check
permutation  w2  r1  p1  w3  p3  r2  c3  c1  p2  c2  check
permutation  w2  r1  p1  w3  p3  c3  r2  p2  c1  c2  check
permutation  w2  r1  p1  w3  p3  c3  r2  p2  c2  c1  check
permutation  w2  r1  p1  w3  p3  c3  r2  c1  p2  c2  check
permutation  w2  r1  p1  w3  p3  c3  c1  r2  p2  c2  check
permutation  w3  r1  r2  w2  p1  p2  p3  c3  c1  c2  check
permutation  w3  r1  r2  w2  p1  p2  p3  c3  c2  c1  check
permutation  w3  r1  r2  w2  p1  p3  p2  c3  c1  c2  check
permutation  w3  r1  r2  w2  p1  p3  p2  c3  c2  c1  check
permutation  w3  r1  r2  w2  p1  p3  c3  p2  c1  c2  check
permutation  w3  r1  r2  w2  p1  p3  c3  p2  c2  c1  check
permutation  w3  r1  r2  w2  p1  p3  c3  c1  p2  c2  check
permutation  w3  r1  r2  w2  p2  p1  p3  c3  c1  c2  check
permutation  w3  r1  r2  w2  p2  p1  p3  c3  c2  c1  check
permutation  w3  r1  r2  w2  p2  p3  p1  c3  c1  c2  check
permutation  w3  r1  r2  w2  p2  p3  p1  c3  c2  c1  check
permutation  w3  r1  r2  w2  p2  p3  c3  p1  c1  c2  check
permutation  w3  r1  r2  w2  p2  p3  c3  p1  c2  c1  check
permutation  w3  r1  r2  w2  p2  p3  c3  c2  p1  c1  check
permutation  w3  r1  r2  w2  p3  p1  p2  c3  c1  c2  check
permutation  w3  r1  r2  w2  p3  p1  p2  c3  c2  c1  check
permutation  w3  r1  r2  w2  p3  p1  c3  p2  c1  c2  check
permutation  w3  r1  r2  w2  p3  p1  c3  p2  c2  c1  check
permutation  w3  r1  r2  w2  p3  p1  c3  c1  p2  c2  check
permutation  w3  r1  r2  w2  p3  p2  p1  c3  c1  c2  check
permutation  w3  r1  r2  w2  p3  p2  p1  c3  c2  c1  check
permutation  w3  r1  r2  w2  p3  p2  c3  p1  c1  c2  check
permutation  w3  r1  r2  w2  p3  p2  c3  p1  c2  c1  check
permutation  w3  r1  r2  w2  p3  p2  c3  c2  p1  c1  check
permutation  w3  r1  r2  w2  p3  c3  p1  p2  c1  c2  check
permutation  w3  r1  r2  w2  p3  c3  p1  p2  c2  c1  check
permutation  w3  r1  r2  w2  p3  c3  p1  c1  p2  c2  check
permutation  w3  r1  r2  w2  p3  c3  p2  p1  c1  c2  check
permutation  w3  r1  r2  w2  p3  c3  p2  p1  c2  c1  check
permutation  w3  r1  r2  w2  p3  c3  p2  c2  p1  c1  check
permutation  w3  r1  r2  p1  w2  p2  p3  c3  c1  c2  check
permutation  w3  r1  r2  p1  w2  p2  p3  c3  c2  c1  check
permutation  w3  r1  r2  p1  w2  p3  p2  c3  c1  c2  check
permutation  w3  r1  r2  p1  w2  p3  p2  c3  c2  c1  check
permutation  w3  r1  r2  p1  w2  p3  c3  p2  c1  c2  check
permutation  w3  r1  r2  p1  w2  p3  c3  p2  c2  c1  check
permutation  w3  r1  r2  p1  w2  p3  c3  c1  p2  c2  check
permutation  w3  r1  r2  p1  p3  w2  p2  c3  c1  c2  check
permutation  w3  r1  r2  p1  p3  w2  p2  c3  c2  c1  check
permutation  w3  r1  r2  p1  p3  w2  c3  p2  c1  c2  check
permutation  w3  r1  r2  p1  p3  w2  c3  p2  c2  c1  check
permutation  w3  r1  r2  p1  p3  w2  c3  c1  p2  c2  check
permutation  w3  r1  r2  p1  p3  c3  w2  p2  c1  c2  check
permutation  w3  r1  r2  p1  p3  c3  w2  p2  c2  c1  check
permutation  w3  r1  r2  p1  p3  c3  w2  c1  p2  c2  check
permutation  w3  r1  r2  p1  p3  c3  c1  w2  p2  c2  check
permutation  w3  r1  r2  p3  w2  p1  p2  c3  c1  c2  check
permutation  w3  r1  r2  p3  w2  p1  p2  c3  c2  c1  check
permutation  w3  r1  r2  p3  w2  p1  c3  p2  c1  c2  check
permutation  w3  r1  r2  p3  w2  p1  c3  p2  c2  c1  check
permutation  w3  r1  r2  p3  w2  p1  c3  c1  p2  c2  check
permutation  w3  r1  r2  p3  w2  p2  p1  c3  c1  c2  check
permutation  w3  r1  r2  p3  w2  p2  p1  c3  c2  c1  check
permutation  w3  r1  r2  p3  w2  p2  c3  p1  c1  c2  check
permutation  w3  r1  r2  p3  w2  p2  c3  p1  c2  c1  check
permutation  w3  r1  r2  p3  w2  p2  c3  c2  p1  c1  check
permutation  w3  r1  r2  p3  w2  c3  p1  p2  c1  c2  check
permutation  w3  r1  r2  p3  w2  c3  p1  p2  c2  c1  check
permutation  w3  r1  r2  p3  w2  c3  p1  c1  p2  c2  check
permutation  w3  r1  r2  p3  w2  c3  p2  p1  c1  c2  check
permutation  w3  r1  r2  p3  w2  c3  p2  p1  c2  c1  check
permutation  w3  r1  r2  p3  w2  c3  p2  c2  p1  c1  check
permutation  w3  r1  r2  p3  p1  w2  p2  c3  c1  c2  check
permutation  w3  r1  r2  p3  p1  w2  p2  c3  c2  c1  check
permutation  w3  r1  r2  p3  p1  w2  c3  p2  c1  c2  check
permutation  w3  r1  r2  p3  p1  w2  c3  p2  c2  c1  check
permutation  w3  r1  r2  p3  p1  w2  c3  c1  p2  c2  check
permutation  w3  r1  r2  p3  p1  c3  w2  p2  c1  c2  check
permutation  w3  r1  r2  p3  p1  c3  w2  p2  c2  c1  check
permutation  w3  r1  r2  p3  p1  c3  w2  c1  p2  c2  check
permutation  w3  r1  r2  p3  p1  c3  c1  w2  p2  c2  check
permutation  w3  r1  r2  p3  c3  w2  p1  p2  c1  c2  check
permutation  w3  r1  r2  p3  c3  w2  p1  p2  c2  c1  check
permutation  w3  r1  r2  p3  c3  w2  p1  c1  p2  c2  check
permutation  w3  r1  r2  p3  c3  w2  p2  p1  c1  c2  check
permutation  w3  r1  r2  p3  c3  w2  p2  p1  c2  c1  check
permutation  w3  r1  r2  p3  c3  w2  p2  c2  p1  c1  check
permutation  w3  r1  r2  p3  c3  p1  w2  p2  c1  c2  check
permutation  w3  r1  r2  p3  c3  p1  w2  p2  c2  c1  check
permutation  w3  r1  r2  p3  c3  p1  w2  c1  p2  c2  check
permutation  w3  r1  r2  p3  c3  p1  c1  w2  p2  c2  check
permutation  w3  r1  w2  r2  p1  p2  p3  c3  c1  c2  check
permutation  w3  r1  w2  r2  p1  p2  p3  c3  c2  c1  check
permutation  w3  r1  w2  r2  p1  p3  p2  c3  c1  c2  check
permutation  w3  r1  w2  r2  p1  p3  p2  c3  c2  c1  check
permutation  w3  r1  w2  r2  p1  p3  c3  p2  c1  c2  check
permutation  w3  r1  w2  r2  p1  p3  c3  p2  c2  c1  check
permutation  w3  r1  w2  r2  p1  p3  c3  c1  p2  c2  check
permutation  w3  r1  w2  r2  p2  p1  p3  c3  c1  c2  check
permutation  w3  r1  w2  r2  p2  p1  p3  c3  c2  c1  check
permutation  w3  r1  w2  r2  p2  p3  p1  c3  c1  c2  check
permutation  w3  r1  w2  r2  p2  p3  p1  c3  c2  c1  check
permutation  w3  r1  w2  r2  p2  p3  c3  p1  c1  c2  check
permutation  w3  r1  w2  r2  p2  p3  c3  p1  c2  c1  check
permutation  w3  r1  w2  r2  p2  p3  c3  c2  p1  c1  check
permutation  w3  r1  w2  r2  p3  p1  p2  c3  c1  c2  check
permutation  w3  r1  w2  r2  p3  p1  p2  c3  c2  c1  check
permutation  w3  r1  w2  r2  p3  p1  c3  p2  c1  c2  check
permutation  w3  r1  w2  r2  p3  p1  c3  p2  c2  c1  check
permutation  w3  r1  w2  r2  p3  p1  c3  c1  p2  c2  check
permutation  w3  r1  w2  r2  p3  p2  p1  c3  c1  c2  check
permutation  w3  r1  w2  r2  p3  p2  p1  c3  c2  c1  check
permutation  w3  r1  w2  r2  p3  p2  c3  p1  c1  c2  check
permutation  w3  r1  w2  r2  p3  p2  c3  p1  c2  c1  check
permutation  w3  r1  w2  r2  p3  p2  c3  c2  p1  c1  check
permutation  w3  r1  w2  r2  p3  c3  p1  p2  c1  c2  check
permutation  w3  r1  w2  r2  p3  c3  p1  p2  c2  c1  check
permutation  w3  r1  w2  r2  p3  c3  p1  c1  p2  c2  check
permutation  w3  r1  w2  r2  p3  c3  p2  p1  c1  c2  check
permutation  w3  r1  w2  r2  p3  c3  p2  p1  c2  c1  check
permutation  w3  r1  w2  r2  p3  c3  p2  c2  p1  c1  check
permutation  w3  r1  w2  p1  r2  p2  p3  c3  c1  c2  check
permutation  w3  r1  w2  p1  r2  p2  p3  c3  c2  c1  check
permutation  w3  r1  w2  p1  r2  p3  p2  c3  c1  c2  check
permutation  w3  r1  w2  p1  r2  p3  p2  c3  c2  c1  check
permutation  w3  r1  w2  p1  r2  p3  c3  p2  c1  c2  check
permutation  w3  r1  w2  p1  r2  p3  c3  p2  c2  c1  check
permutation  w3  r1  w2  p1  r2  p3  c3  c1  p2  c2  check
permutation  w3  r1  w2  p1  p3  r2  p2  c3  c1  c2  check
permutation  w3  r1  w2  p1  p3  r2  p2  c3  c2  c1  check
permutation  w3  r1  w2  p1  p3  r2  c3  p2  c1  c2  check
permutation  w3  r1  w2  p1  p3  r2  c3  p2  c2  c1  check
permutation  w3  r1  w2  p1  p3  r2  c3  c1  p2  c2  check
permutation  w3  r1  w2  p1  p3  c3  r2  p2  c1  c2  check
permutation  w3  r1  w2  p1  p3  c3  r2  p2  c2  c1  check
permutation  w3  r1  w2  p1  p3  c3  r2  c1  p2  c2  check
permutation  w3  r1  w2  p1  p3  c3  c1  r2  p2  c2  check
permutation  w3  r1  w2  p3  r2  p1  p2  c3  c1  c2  check
permutation  w3  r1  w2  p3  r2  p1  p2  c3  c2  c1  check
permutation  w3  r1  w2  p3  r2  p1  c3  p2  c1  c2  check
permutation  w3  r1  w2  p3  r2  p1  c3  p2  c2  c1  check
permutation  w3  r1  w2  p3  r2  p1  c3  c1  p2  c2  check
permutation  w3  r1  w2  p3  r2  p2  p1  c3  c1  c2  check
permutation  w3  r1  w2  p3  r2  p2  p1  c3  c2  c1  check
permutation  w3  r1  w2  p3  r2  p2  c3  p1  c1  c2  check
permutation  w3  r1  w2  p3  r2  p2  c3  p1  c2  c1  check
permutation  w3  r1  w2  p3  r2  p2  c3  c2  p1  c1  check
permutation  w3  r1  w2  p3  r2  c3  p1  p2  c1  c2  check
permutation  w3  r1  w2  p3  r2  c3  p1  p2  c2  c1  check
permutation  w3  r1  w2  p3  r2  c3  p1  c1  p2  c2  check
permutation  w3  r1  w2  p3  r2  c3  p2  p1  c1  c2  check
permutation  w3  r1  w2  p3  r2  c3  p2  p1  c2  c1  check
permutation  w3  r1  w2  p3  r2  c3  p2  c2  p1  c1  check
permutation  w3  r1  w2  p3  p1  r2  p2  c3  c1  c2  check
permutation  w3  r1  w2  p3  p1  r2  p2  c3  c2  c1  check
permutation  w3  r1  w2  p3  p1  r2  c3  p2  c1  c2  check
permutation  w3  r1  w2  p3  p1  r2  c3  p2  c2  c1  check
permutation  w3  r1  w2  p3  p1  r2  c3  c1  p2  c2  check
permutation  w3  r1  w2  p3  p1  c3  r2  p2  c1  c2  check
permutation  w3  r1  w2  p3  p1  c3  r2  p2  c2  c1  check
permutation  w3  r1  w2  p3  p1  c3  r2  c1  p2  c2  check
permutation  w3  r1  w2  p3  p1  c3  c1  r2  p2  c2  check
permutation  w3  r1  w2  p3  c3  r2  p1  p2  c1  c2  check
permutation  w3  r1  w2  p3  c3  r2  p1  p2  c2  c1  check
permutation  w3  r1  w2  p3  c3  r2  p1  c1  p2  c2  check
permutation  w3  r1  w2  p3  c3  r2  p2  p1  c1  c2  check
permutation  w3  r1  w2  p3  c3  r2  p2  p1  c2  c1  check
permutation  w3  r1  w2  p3  c3  r2  p2  c2  p1  c1  check
permutation  w3  r1  w2  p3  c3  p1  r2  p2  c1  c2  check
permutation  w3  r1  w2  p3  c3  p1  r2  p2  c2  c1  check
permutation  w3  r1  w2  p3  c3  p1  r2  c1  p2  c2  check
permutation  w3  r1  w2  p3  c3  p1  c1  r2  p2  c2  check
permutation  w3  r1  p1  r2  w2  p2  p3  c3  c1  c2  check
permutation  w3  r1  p1  r2  w2  p2  p3  c3  c2  c1  check
permutation  w3  r1  p1  r2  w2  p3  p2  c3  c1  c2  check
permutation  w3  r1  p1  r2  w2  p3  p2  c3  c2  c1  check
permutation  w3  r1  p1  r2  w2  p3  c3  p2  c1  c2  check
permutation  w3  r1  p1  r2  w2  p3  c3  p2  c2  c1  check
permutation  w3  r1  p1  r2  w2  p3  c3  c1  p2  c2  check
permutation  w3  r1  p1  r2  p3  w2  p2  c3  c1  c2  check
permutation  w3  r1  p1  r2  p3  w2  p2  c3  c2  c1  check
permutation  w3  r1  p1  r2  p3  w2  c3  p2  c1  c2  check
permutation  w3  r1  p1  r2  p3  w2  c3  p2  c2  c1  check
permutation  w3  r1  p1  r2  p3  w2  c3  c1  p2  c2  check
permutation  w3  r1  p1  r2  p3  c3  w2  p2  c1  c2  check
permutation  w3  r1  p1  r2  p3  c3  w2  p2  c2  c1  check
permutation  w3  r1  p1  r2  p3  c3  w2  c1  p2  c2  check
permutation  w3  r1  p1  r2  p3  c3  c1  w2  p2  c2  check
permutation  w3  r1  p1  w2  r2  p2  p3  c3  c1  c2  check
permutation  w3  r1  p1  w2  r2  p2  p3  c3  c2  c1  check
permutation  w3  r1  p1  w2  r2  p3  p2  c3  c1  c2  check
permutation  w3  r1  p1  w2  r2  p3  p2  c3  c2  c1  check
permutation  w3  r1  p1  w2  r2  p3  c3  p2  c1  c2  check
permutation  w3  r1  p1  w2  r2  p3  c3  p2  c2  c1  check
permutation  w3  r1  p1  w2  r2  p3  c3  c1  p2  c2  check
permutation  w3  r1  p1  w2  p3  r2  p2  c3  c1  c2  check
permutation  w3  r1  p1  w2  p3  r2  p2  c3  c2  c1  check
permutation  w3  r1  p1  w2  p3  r2  c3  p2  c1  c2  check
permutation  w3  r1  p1  w2  p3  r2  c3  p2  c2  c1  check
permutation  w3  r1  p1  w2  p3  r2  c3  c1  p2  c2  check
permutation  w3  r1  p1  w2  p3  c3  r2  p2  c1  c2  check
permutation  w3  r1  p1  w2  p3  c3  r2  p2  c2  c1  check
permutation  w3  r1  p1  w2  p3  c3  r2  c1  p2  c2  check
permutation  w3  r1  p1  w2  p3  c3  c1  r2  p2  c2  check
permutation  w3  r1  p1  p3  r2  w2  p2  c3  c1  c2  check
permutation  w3  r1  p1  p3  r2  w2  p2  c3  c2  c1  check
permutation  w3  r1  p1  p3  r2  w2  c3  p2  c1  c2  check
permutation  w3  r1  p1  p3  r2  w2  c3  p2  c2  c1  check
permutation  w3  r1  p1  p3  r2  w2  c3  c1  p2  c2  check
permutation  w3  r1  p1  p3  r2  c3  w2  p2  c1  c2  check
permutation  w3  r1  p1  p3  r2  c3  w2  p2  c2  c1  check
permutation  w3  r1  p1  p3  r2  c3  w2  c1  p2  c2  check
permutation  w3  r1  p1  p3  r2  c3  c1  w2  p2  c2  check
permutation  w3  r1  p1  p3  w2  r2  p2  c3  c1  c2  check
permutation  w3  r1  p1  p3  w2  r2  p2  c3  c2  c1  check
permutation  w3  r1  p1  p3  w2  r2  c3  p2  c1  c2  check
permutation  w3  r1  p1  p3  w2  r2  c3  p2  c2  c1  check
permutation  w3  r1  p1  p3  w2  r2  c3  c1  p2  c2  check
permutation  w3  r1  p1  p3  w2  c3  r2  p2  c1  c2  check
permutation  w3  r1  p1  p3  w2  c3  r2  p2  c2  c1  check
permutation  w3  r1  p1  p3  w2  c3  r2  c1  p2  c2  check
permutation  w3  r1  p1  p3  w2  c3  c1  r2  p2  c2  check
permutation  w3  r1  p1  p3  c3  r2  w2  p2  c1  c2  check
permutation  w3  r1  p1  p3  c3  r2  w2  p2  c2  c1  check
permutation  w3  r1  p1  p3  c3  r2  w2  c1  p2  c2  check
permutation  w3  r1  p1  p3  c3  r2  c1  w2  p2  c2  check
permutation  w3  r1  p1  p3  c3  w2  r2  p2  c1  c2  check
permutation  w3  r1  p1  p3  c3  w2  r2  p2  c2  c1  check
permutation  w3  r1  p1  p3  c3  w2  r2  c1  p2  c2  check
permutation  w3  r1  p1  p3  c3  w2  c1  r2  p2  c2  check
permutation  w3  r1  p1  p3  c3  c1  r2  w2  p2  c2  check
permutation  w3  r1  p1  p3  c3  c1  w2  r2  p2  c2  check
permutation  w3  r1  p3  r2  w2  p1  p2  c3  c1  c2  check
permutation  w3  r1  p3  r2  w2  p1  p2  c3  c2  c1  check
permutation  w3  r1  p3  r2  w2  p1  c3  p2  c1  c2  check
permutation  w3  r1  p3  r2  w2  p1  c3  p2  c2  c1  check
permutation  w3  r1  p3  r2  w2  p1  c3  c1  p2  c2  check
permutation  w3  r1  p3  r2  w2  p2  p1  c3  c1  c2  check
permutation  w3  r1  p3  r2  w2  p2  p1  c3  c2  c1  check
permutation  w3  r1  p3  r2  w2  p2  c3  p1  c1  c2  check
permutation  w3  r1  p3  r2  w2  p2  c3  p1  c2  c1  check
permutation  w3  r1  p3  r2  w2  p2  c3  c2  p1  c1  check
permutation  w3  r1  p3  r2  w2  c3  p1  p2  c1  c2  check
permutation  w3  r1  p3  r2  w2  c3  p1  p2  c2  c1  check
permutation  w3  r1  p3  r2  w2  c3  p1  c1  p2  c2  check
permutation  w3  r1  p3  r2  w2  c3  p2  p1  c1  c2  check
permutation  w3  r1  p3  r2  w2  c3  p2  p1  c2  c1  check
permutation  w3  r1  p3  r2  w2  c3  p2  c2  p1  c1  check
permutation  w3  r1  p3  r2  p1  w2  p2  c3  c1  c2  check
permutation  w3  r1  p3  r2  p1  w2  p2  c3  c2  c1  check
permutation  w3  r1  p3  r2  p1  w2  c3  p2  c1  c2  check
permutation  w3  r1  p3  r2  p1  w2  c3  p2  c2  c1  check
permutation  w3  r1  p3  r2  p1  w2  c3  c1  p2  c2  check
permutation  w3  r1  p3  r2  p1  c3  w2  p2  c1  c2  check
permutation  w3  r1  p3  r2  p1  c3  w2  p2  c2  c1  check
permutation  w3  r1  p3  r2  p1  c3  w2  c1  p2  c2  check
permutation  w3  r1  p3  r2  p1  c3  c1  w2  p2  c2  check
permutation  w3  r1  p3  r2  c3  w2  p1  p2  c1  c2  check
permutation  w3  r1  p3  r2  c3  w2  p1  p2  c2  c1  check
permutation  w3  r1  p3  r2  c3  w2  p1  c1  p2  c2  check
permutation  w3  r1  p3  r2  c3  w2  p2  p1  c1  c2  check
permutation  w3  r1  p3  r2  c3  w2  p2  p1  c2  c1  check
permutation  w3  r1  p3  r2  c3  w2  p2  c2  p1  c1  check
permutation  w3  r1  p3  r2  c3  p1  w2  p2  c1  c2  check
permutation  w3  r1  p3  r2  c3  p1  w2  p2  c2  c1  check
permutation  w3  r1  p3  r2  c3  p1  w2  c1  p2  c2  check
permutation  w3  r1  p3  r2  c3  p1  c1  w2  p2  c2  check
permutation  w3  r1  p3  w2  r2  p1  p2  c3  c1  c2  check
permutation  w3  r1  p3  w2  r2  p1  p2  c3  c2  c1  check
permutation  w3  r1  p3  w2  r2  p1  c3  p2  c1  c2  check
permutation  w3  r1  p3  w2  r2  p1  c3  p2  c2  c1  check
permutation  w3  r1  p3  w2  r2  p1  c3  c1  p2  c2  check
permutation  w3  r1  p3  w2  r2  p2  p1  c3  c1  c2  check
permutation  w3  r1  p3  w2  r2  p2  p1  c3  c2  c1  check
permutation  w3  r1  p3  w2  r2  p2  c3  p1  c1  c2  check
permutation  w3  r1  p3  w2  r2  p2  c3  p1  c2  c1  check
permutation  w3  r1  p3  w2  r2  p2  c3  c2  p1  c1  check
permutation  w3  r1  p3  w2  r2  c3  p1  p2  c1  c2  check
permutation  w3  r1  p3  w2  r2  c3  p1  p2  c2  c1  check
permutation  w3  r1  p3  w2  r2  c3  p1  c1  p2  c2  check
permutation  w3  r1  p3  w2  r2  c3  p2  p1  c1  c2  check
permutation  w3  r1  p3  w2  r2  c3  p2  p1  c2  c1  check
permutation  w3  r1  p3  w2  r2  c3  p2  c2  p1  c1  check
permutation  w3  r1  p3  w2  p1  r2  p2  c3  c1  c2  check
permutation  w3  r1  p3  w2  p1  r2  p2  c3  c2  c1  check
permutation  w3  r1  p3  w2  p1  r2  c3  p2  c1  c2  check
permutation  w3  r1  p3  w2  p1  r2  c3  p2  c2  c1  check
permutation  w3  r1  p3  w2  p1  r2  c3  c1  p2  c2  check
permutation  w3  r1  p3  w2  p1  c3  r2  p2  c1  c2  check
permutation  w3  r1  p3  w2  p1  c3  r2  p2  c2  c1  check
permutation  w3  r1  p3  w2  p1  c3  r2  c1  p2  c2  check
permutation  w3  r1  p3  w2  p1  c3  c1  r2  p2  c2  check
permutation  w3  r1  p3  w2  c3  r2  p1  p2  c1  c2  check
permutation  w3  r1  p3  w2  c3  r2  p1  p2  c2  c1  check
permutation  w3  r1  p3  w2  c3  r2  p1  c1  p2  c2  check
permutation  w3  r1  p3  w2  c3  r2  p2  p1  c1  c2  check
permutation  w3  r1  p3  w2  c3  r2  p2  p1  c2  c1  check
permutation  w3  r1  p3  w2  c3  r2  p2  c2  p1  c1  check
permutation  w3  r1  p3  w2  c3  p1  r2  p2  c1  c2  check
permutation  w3  r1  p3  w2  c3  p1  r2  p2  c2  c1  check
permutation  w3  r1  p3  w2  c3  p1  r2  c1  p2  c2  check
permutation  w3  r1  p3  w2  c3  p1  c1  r2  p2  c2  check
permutation  w3  r1  p3  p1  r2  w2  p2  c3  c1  c2  check
permutation  w3  r1  p3  p1  r2  w2  p2  c3  c2  c1  check
permutation  w3  r1  p3  p1  r2  w2  c3  p2  c1  c2  check
permutation  w3  r1  p3  p1  r2  w2  c3  p2  c2  c1  check
permutation  w3  r1  p3  p1  r2  w2  c3  c1  p2  c2  check
permutation  w3  r1  p3  p1  r2  c3  w2  p2  c1  c2  check
permutation  w3  r1  p3  p1  r2  c3  w2  p2  c2  c1  check
permutation  w3  r1  p3  p1  r2  c3  w2  c1  p2  c2  check
permutation  w3  r1  p3  p1  r2  c3  c1  w2  p2  c2  check
permutation  w3  r1  p3  p1  w2  r2  p2  c3  c1  c2  check
permutation  w3  r1  p3  p1  w2  r2  p2  c3  c2  c1  check
permutation  w3  r1  p3  p1  w2  r2  c3  p2  c1  c2  check
permutation  w3  r1  p3  p1  w2  r2  c3  p2  c2  c1  check
permutation  w3  r1  p3  p1  w2  r2  c3  c1  p2  c2  check
permutation  w3  r1  p3  p1  w2  c3  r2  p2  c1  c2  check
permutation  w3  r1  p3  p1  w2  c3  r2  p2  c2  c1  check
permutation  w3  r1  p3  p1  w2  c3  r2  c1  p2  c2  check
permutation  w3  r1  p3  p1  w2  c3  c1  r2  p2  c2  check
permutation  w3  r1  p3  p1  c3  r2  w2  p2  c1  c2  check
permutation  w3  r1  p3  p1  c3  r2  w2  p2  c2  c1  check
permutation  w3  r1  p3  p1  c3  r2  w2  c1  p2  c2  check
permutation  w3  r1  p3  p1  c3  r2  c1  w2  p2  c2  check
permutation  w3  r1  p3  p1  c3  w2  r2  p2  c1  c2  check
permutation  w3  r1  p3  p1  c3  w2  r2  p2  c2  c1  check
permutation  w3  r1  p3  p1  c3  w2  r2  c1  p2  c2  check
permutation  w3  r1  p3  p1  c3  w2  c1  r2  p2  c2  check
permutation  w3  r1  p3  p1  c3  c1  r2  w2  p2  c2  check
permutation  w3  r1  p3  p1  c3  c1  w2  r2  p2  c2  check
permutation  w3  r1  p3  c3  r2  w2  p1  p2  c1  c2  check
permutation  w3  r1  p3  c3  r2  w2  p1  p2  c2  c1  check
permutation  w3  r1  p3  c3  r2  w2  p1  c1  p2  c2  check
permutation  w3  r1  p3  c3  r2  w2  p2  p1  c1  c2  check
permutation  w3  r1  p3  c3  r2  w2  p2  p1  c2  c1  check
permutation  w3  r1  p3  c3  r2  w2  p2  c2  p1  c1  check
permutation  w3  r1  p3  c3  r2  p1  w2  p2  c1  c2  check
permutation  w3  r1  p3  c3  r2  p1  w2  p2  c2  c1  check
permutation  w3  r1  p3  c3  r2  p1  w2  c1  p2  c2  check
permutation  w3  r1  p3  c3  r2  p1  c1  w2  p2  c2  check
permutation  w3  r1  p3  c3  w2  r2  p1  p2  c1  c2  check
permutation  w3  r1  p3  c3  w2  r2  p1  p2  c2  c1  check
permutation  w3  r1  p3  c3  w2  r2  p1  c1  p2  c2  check
permutation  w3  r1  p3  c3  w2  r2  p2  p1  c1  c2  check
permutation  w3  r1  p3  c3  w2  r2  p2  p1  c2  c1  check
permutation  w3  r1  p3  c3  w2  r2  p2  c2  p1  c1  check
permutation  w3  r1  p3  c3  w2  p1  r2  p2  c1  c2  check
permutation  w3  r1  p3  c3  w2  p1  r2  p2  c2  c1  check
permutation  w3  r1  p3  c3  w2  p1  r2  c1  p2  c2  check
permutation  w3  r1  p3  c3  w2  p1  c1  r2  p2  c2  check
permutation  w3  r1  p3  c3  p1  r2  w2  p2  c1  c2  check
permutation  w3  r1  p3  c3  p1  r2  w2  p2  c2  c1  check
permutation  w3  r1  p3  c3  p1  r2  w2  c1  p2  c2  check
permutation  w3  r1  p3  c3  p1  r2  c1  w2  p2  c2  check
permutation  w3  r1  p3  c3  p1  w2  r2  p2  c1  c2  check
permutation  w3  r1  p3  c3  p1  w2  r2  p2  c2  c1  check
permutation  w3  r1  p3  c3  p1  w2  r2  c1  p2  c2  check
permutation  w3  r1  p3  c3  p1  w2  c1  r2  p2  c2  check
permutation  w3  r1  p3  c3  p1  c1  r2  w2  p2  c2  check
permutation  w3  r1  p3  c3  p1  c1  w2  r2  p2  c2  check
permutation  w3  r2  r1  w2  p1  p2  p3  c3  c1  c2  check
permutation  w3  r2  r1  w2  p1  p2  p3  c3  c2  c1  check
permutation  w3  r2  r1  w2  p1  p3  p2  c3  c1  c2  check
permutation  w3  r2  r1  w2  p1  p3  p2  c3  c2  c1  check
permutation  w3  r2  r1  w2  p1  p3  c3  p2  c1  c2  check
permutation  w3  r2  r1  w2  p1  p3  c3  p2  c2  c1  check
permutation  w3  r2  r1  w2  p1  p3  c3  c1  p2  c2  check
permutation  w3  r2  r1  w2  p2  p1  p3  c3  c1  c2  check
permutation  w3  r2  r1  w2  p2  p1  p3  c3  c2  c1  check
permutation  w3  r2  r1  w2  p2  p3  p1  c3  c1  c2  check
permutation  w3  r2  r1  w2  p2  p3  p1  c3  c2  c1  check
permutation  w3  r2  r1  w2  p2  p3  c3  p1  c1  c2  check
permutation  w3  r2  r1  w2  p2  p3  c3  p1  c2  c1  check
permutation  w3  r2  r1  w2  p2  p3  c3  c2  p1  c1  check
permutation  w3  r2  r1  w2  p3  p1  p2  c3  c1  c2  check
permutation  w3  r2  r1  w2  p3  p1  p2  c3  c2  c1  check
permutation  w3  r2  r1  w2  p3  p1  c3  p2  c1  c2  check
permutation  w3  r2  r1  w2  p3  p1  c3  p2  c2  c1  check
permutation  w3  r2  r1  w2  p3  p1  c3  c1  p2  c2  check
permutation  w3  r2  r1  w2  p3  p2  p1  c3  c1  c2  check
permutation  w3  r2  r1  w2  p3  p2  p1  c3  c2  c1  check
permutation  w3  r2  r1  w2  p3  p2  c3  p1  c1  c2  check
permutation  w3  r2  r1  w2  p3  p2  c3  p1  c2  c1  check
permutation  w3  r2  r1  w2  p3  p2  c3  c2  p1  c1  check
permutation  w3  r2  r1  w2  p3  c3  p1  p2  c1  c2  check
permutation  w3  r2  r1  w2  p3  c3  p1  p2  c2  c1  check
permutation  w3  r2  r1  w2  p3  c3  p1  c1  p2  c2  check
permutation  w3  r2  r1  w2  p3  c3  p2  p1  c1  c2  check
permutation  w3  r2  r1  w2  p3  c3  p2  p1  c2  c1  check
permutation  w3  r2  r1  w2  p3  c3  p2  c2  p1  c1  check
permutation  w3  r2  r1  p1  w2  p2  p3  c3  c1  c2  check
permutation  w3  r2  r1  p1  w2  p2  p3  c3  c2  c1  check
permutation  w3  r2  r1  p1  w2  p3  p2  c3  c1  c2  check
permutation  w3  r2  r1  p1  w2  p3  p2  c3  c2  c1  check
permutation  w3  r2  r1  p1  w2  p3  c3  p2  c1  c2  check
permutation  w3  r2  r1  p1  w2  p3  c3  p2  c2  c1  check
permutation  w3  r2  r1  p1  w2  p3  c3  c1  p2  c2  check
permutation  w3  r2  r1  p1  p3  w2  p2  c3  c1  c2  check
permutation  w3  r2  r1  p1  p3  w2  p2  c3  c2  c1  check
permutation  w3  r2  r1  p1  p3  w2  c3  p2  c1  c2  check
permutation  w3  r2  r1  p1  p3  w2  c3  p2  c2  c1  check
permutation  w3  r2  r1  p1  p3  w2  c3  c1  p2  c2  check
permutation  w3  r2  r1  p1  p3  c3  w2  p2  c1  c2  check
permutation  w3  r2  r1  p1  p3  c3  w2  p2  c2  c1  check
permutation  w3  r2  r1  p1  p3  c3  w2  c1  p2  c2  check
permutation  w3  r2  r1  p1  p3  c3  c1  w2  p2  c2  check
permutation  w3  r2  r1  p3  w2  p1  p2  c3  c1  c2  check
permutation  w3  r2  r1  p3  w2  p1  p2  c3  c2  c1  check
permutation  w3  r2  r1  p3  w2  p1  c3  p2  c1  c2  check
permutation  w3  r2  r1  p3  w2  p1  c3  p2  c2  c1  check
permutation  w3  r2  r1  p3  w2  p1  c3  c1  p2  c2  check
permutation  w3  r2  r1  p3  w2  p2  p1  c3  c1  c2  check
permutation  w3  r2  r1  p3  w2  p2  p1  c3  c2  c1  check
permutation  w3  r2  r1  p3  w2  p2  c3  p1  c1  c2  check
permutation  w3  r2  r1  p3  w2  p2  c3  p1  c2  c1  check
permutation  w3  r2  r1  p3  w2  p2  c3  c2  p1  c1  check
permutation  w3  r2  r1  p3  w2  c3  p1  p2  c1  c2  check
permutation  w3  r2  r1  p3  w2  c3  p1  p2  c2  c1  check
permutation  w3  r2  r1  p3  w2  c3  p1  c1  p2  c2  check
permutation  w3  r2  r1  p3  w2  c3  p2  p1  c1  c2  check
permutation  w3  r2  r1  p3  w2  c3  p2  p1  c2  c1  check
permutation  w3  r2  r1  p3  w2  c3  p2  c2  p1  c1  check
permutation  w3  r2  r1  p3  p1  w2  p2  c3  c1  c2  check
permutation  w3  r2  r1  p3  p1  w2  p2  c3  c2  c1  check
permutation  w3  r2  r1  p3  p1  w2  c3  p2  c1  c2  check
permutation  w3  r2  r1  p3  p1  w2  c3  p2  c2  c1  check
permutation  w3  r2  r1  p3  p1  w2  c3  c1  p2  c2  check
permutation  w3  r2  r1  p3  p1  c3  w2  p2  c1  c2  check
permutation  w3  r2  r1  p3  p1  c3  w2  p2  c2  c1  check
permutation  w3  r2  r1  p3  p1  c3  w2  c1  p2  c2  check
permutation  w3  r2  r1  p3  p1  c3  c1  w2  p2  c2  check
permutation  w3  r2  r1  p3  c3  w2  p1  p2  c1  c2  check
permutation  w3  r2  r1  p3  c3  w2  p1  p2  c2  c1  check
permutation  w3  r2  r1  p3  c3  w2  p1  c1  p2  c2  check
permutation  w3  r2  r1  p3  c3  w2  p2  p1  c1  c2  check
permutation  w3  r2  r1  p3  c3  w2  p2  p1  c2  c1  check
permutation  w3  r2  r1  p3  c3  w2  p2  c2  p1  c1  check
permutation  w3  r2  r1  p3  c3  p1  w2  p2  c1  c2  check
permutation  w3  r2  r1  p3  c3  p1  w2  p2  c2  c1  check
permutation  w3  r2  r1  p3  c3  p1  w2  c1  p2  c2  check
permutation  w3  r2  r1  p3  c3  p1  c1  w2  p2  c2  check
permutation  w3  r2  p3  r1  w2  p1  p2  c3  c1  c2  check
permutation  w3  r2  p3  r1  w2  p1  p2  c3  c2  c1  check
permutation  w3  r2  p3  r1  w2  p1  c3  p2  c1  c2  check
permutation  w3  r2  p3  r1  w2  p1  c3  p2  c2  c1  check
permutation  w3  r2  p3  r1  w2  p1  c3  c1  p2  c2  check
permutation  w3  r2  p3  r1  w2  p2  p1  c3  c1  c2  check
permutation  w3  r2  p3  r1  w2  p2  p1  c3  c2  c1  check
permutation  w3  r2  p3  r1  w2  p2  c3  p1  c1  c2  check
permutation  w3  r2  p3  r1  w2  p2  c3  p1  c2  c1  check
permutation  w3  r2  p3  r1  w2  p2  c3  c2  p1  c1  check
permutation  w3  r2  p3  r1  w2  c3  p1  p2  c1  c2  check
permutation  w3  r2  p3  r1  w2  c3  p1  p2  c2  c1  check
permutation  w3  r2  p3  r1  w2  c3  p1  c1  p2  c2  check
permutation  w3  r2  p3  r1  w2  c3  p2  p1  c1  c2  check
permutation  w3  r2  p3  r1  w2  c3  p2  p1  c2  c1  check
permutation  w3  r2  p3  r1  w2  c3  p2  c2  p1  c1  check
permutation  w3  r2  p3  r1  p1  w2  p2  c3  c1  c2  check
permutation  w3  r2  p3  r1  p1  w2  p2  c3  c2  c1  check
permutation  w3  r2  p3  r1  p1  w2  c3  p2  c1  c2  check
permutation  w3  r2  p3  r1  p1  w2  c3  p2  c2  c1  check
permutation  w3  r2  p3  r1  p1  w2  c3  c1  p2  c2  check
permutation  w3  r2  p3  r1  p1  c3  w2  p2  c1  c2  check
permutation  w3  r2  p3  r1  p1  c3  w2  p2  c2  c1  check
permutation  w3  r2  p3  r1  p1  c3  w2  c1  p2  c2  check
permutation  w3  r2  p3  r1  p1  c3  c1  w2  p2  c2  check
permutation  w3  r2  p3  r1  c3  w2  p1  p2  c1  c2  check
permutation  w3  r2  p3  r1  c3  w2  p1  p2  c2  c1  check
permutation  w3  r2  p3  r1  c3  w2  p1  c1  p2  c2  check
permutation  w3  r2  p3  r1  c3  w2  p2  p1  c1  c2  check
permutation  w3  r2  p3  r1  c3  w2  p2  p1  c2  c1  check
permutation  w3  r2  p3  r1  c3  w2  p2  c2  p1  c1  check
permutation  w3  r2  p3  r1  c3  p1  w2  p2  c1  c2  check
permutation  w3  r2  p3  r1  c3  p1  w2  p2  c2  c1  check
permutation  w3  r2  p3  r1  c3  p1  w2  c1  p2  c2  check
permutation  w3  r2  p3  r1  c3  p1  c1  w2  p2  c2  check
permutation  w3  r2  p3  c3  r1  w2  p1  p2  c1  c2  check
permutation  w3  r2  p3  c3  r1  w2  p1  p2  c2  c1  check
permutation  w3  r2  p3  c3  r1  w2  p1  c1  p2  c2  check
permutation  w3  r2  p3  c3  r1  w2  p2  p1  c1  c2  check
permutation  w3  r2  p3  c3  r1  w2  p2  p1  c2  c1  check
permutation  w3  r2  p3  c3  r1  w2  p2  c2  p1  c1  check
permutation  w3  r2  p3  c3  r1  p1  w2  p2  c1  c2  check
permutation  w3  r2  p3  c3  r1  p1  w2  p2  c2  c1  check
permutation  w3  r2  p3  c3  r1  p1  w2  c1  p2  c2  check
permutation  w3  r2  p3  c3  r1  p1  c1  w2  p2  c2  check
permutation  w3  p3  r1  r2  w2  p1  p2  c3  c1  c2  check
permutation  w3  p3  r1  r2  w2  p1  p2  c3  c2  c1  check
permutation  w3  p3  r1  r2  w2  p1  c3  p2  c1  c2  check
permutation  w3  p3  r1  r2  w2  p1  c3  p2  c2  c1  check
permutation  w3  p3  r1  r2  w2  p1  c3  c1  p2  c2  check
permutation  w3  p3  r1  r2  w2  p2  p1  c3  c1  c2  check
permutation  w3  p3  r1  r2  w2  p2  p1  c3  c2  c1  check
permutation  w3  p3  r1  r2  w2  p2  c3  p1  c1  c2  check
permutation  w3  p3  r1  r2  w2  p2  c3  p1  c2  c1  check
permutation  w3  p3  r1  r2  w2  p2  c3  c2  p1  c1  check
permutation  w3  p3  r1  r2  w2  c3  p1  p2  c1  c2  check
permutation  w3  p3  r1  r2  w2  c3  p1  p2  c2  c1  check
permutation  w3  p3  r1  r2  w2  c3  p1  c1  p2  c2  check
permutation  w3  p3  r1  r2  w2  c3  p2  p1  c1  c2  check
permutation  w3  p3  r1  r2  w2  c3  p2  p1  c2  c1  check
permutation  w3  p3  r1  r2  w2  c3  p2  c2  p1  c1  check
permutation  w3  p3  r1  r2  p1  w2  p2  c3  c1  c2  check
permutation  w3  p3  r1  r2  p1  w2  p2  c3  c2  c1  check
permutation  w3  p3  r1  r2  p1  w2  c3  p2  c1  c2  check
permutation  w3  p3  r1  r2  p1  w2  c3  p2  c2  c1  check
permutation  w3  p3  r1  r2  p1  w2  c3  c1  p2  c2  check
permutation  w3  p3  r1  r2  p1  c3  w2  p2  c1  c2  check
permutation  w3  p3  r1  r2  p1  c3  w2  p2  c2  c1  check
permutation  w3  p3  r1  r2  p1  c3  w2  c1  p2  c2  check
permutation  w3  p3  r1  r2  p1  c3  c1  w2  p2  c2  check
permutation  w3  p3  r1  r2  c3  w2  p1  p2  c1  c2  check
permutation  w3  p3  r1  r2  c3  w2  p1  p2  c2  c1  check
permutation  w3  p3  r1  r2  c3  w2  p1  c1  p2  c2  check
permutation  w3  p3  r1  r2  c3  w2  p2  p1  c1  c2  check
permutation  w3  p3  r1  r2  c3  w2  p2  p1  c2  c1  check
permutation  w3  p3  r1  r2  c3  w2  p2  c2  p1  c1  check
permutation  w3  p3  r1  r2  c3  p1  w2  p2  c1  c2  check
permutation  w3  p3  r1  r2  c3  p1  w2  p2  c2  c1  check
permutation  w3  p3  r1  r2  c3  p1  w2  c1  p2  c2  check
permutation  w3  p3  r1  r2  c3  p1  c1  w2  p2  c2  check
permutation  w3  p3  r1  w2  r2  p1  p2  c3  c1  c2  check
permutation  w3  p3  r1  w2  r2  p1  p2  c3  c2  c1  check
permutation  w3  p3  r1  w2  r2  p1  c3  p2  c1  c2  check
permutation  w3  p3  r1  w2  r2  p1  c3  p2  c2  c1  check
permutation  w3  p3  r1  w2  r2  p1  c3  c1  p2  c2  check
permutation  w3  p3  r1  w2  r2  p2  p1  c3  c1  c2  check
permutation  w3  p3  r1  w2  r2  p2  p1  c3  c2  c1  check
permutation  w3  p3  r1  w2  r2  p2  c3  p1  c1  c2  check
permutation  w3  p3  r1  w2  r2  p2  c3  p1  c2  c1  check
permutation  w3  p3  r1  w2  r2  p2  c3  c2  p1  c1  check
permutation  w3  p3  r1  w2  r2  c3  p1  p2  c1  c2  check
permutation  w3  p3  r1  w2  r2  c3  p1  p2  c2  c1  check
permutation  w3  p3  r1  w2  r2  c3  p1  c1  p2  c2  check
permutation  w3  p3  r1  w2  r2  c3  p2  p1  c1  c2  check
permutation  w3  p3  r1  w2  r2  c3  p2  p1  c2  c1  check
permutation  w3  p3  r1  w2  r2  c3  p2  c2  p1  c1  check
permutation  w3  p3  r1  w2  p1  r2  p2  c3  c1  c2  check
permutation  w3  p3  r1  w2  p1  r2  p2  c3  c2  c1  check
permutation  w3  p3  r1  w2  p1  r2  c3  p2  c1  c2  check
permutation  w3  p3  r1  w2  p1  r2  c3  p2  c2  c1  check
permutation  w3  p3  r1  w2  p1  r2  c3  c1  p2  c2  check
permutation  w3  p3  r1  w2  p1  c3  r2  p2  c1  c2  check
permutation  w3  p3  r1  w2  p1  c3  r2  p2  c2  c1  check
permutation  w3  p3  r1  w2  p1  c3  r2  c1  p2  c2  check
permutation  w3  p3  r1  w2  p1  c3  c1  r2  p2  c2  check
permutation  w3  p3  r1  w2  c3  r2  p1  p2  c1  c2  check
permutation  w3  p3  r1  w2  c3  r2  p1  p2  c2  c1  check
permutation  w3  p3  r1  w2  c3  r2  p1  c1  p2  c2  check
permutation  w3  p3  r1  w2  c3  r2  p2  p1  c1  c2  check
permutation  w3  p3  r1  w2  c3  r2  p2  p1  c2  c1  check
permutation  w3  p3  r1  w2  c3  r2  p2  c2  p1  c1  check
permutation  w3  p3  r1  w2  c3  p1  r2  p2  c1  c2  check
permutation  w3  p3  r1  w2  c3  p1  r2  p2  c2  c1  check
permutation  w3  p3  r1  w2  c3  p1  r2  c1  p2  c2  check
permutation  w3  p3  r1  w2  c3  p1  c1  r2  p2  c2  check
permutation  w3  p3  r1  p1  r2  w2  p2  c3  c1  c2  check
permutation  w3  p3  r1  p1  r2  w2  p2  c3  c2  c1  check
permutation  w3  p3  r1  p1  r2  w2  c3  p2  c1  c2  check
permutation  w3  p3  r1  p1  r2  w2  c3  p2  c2  c1  check
permutation  w3  p3  r1  p1  r2  w2  c3  c1  p2  c2  check
permutation  w3  p3  r1  p1  r2  c3  w2  p2  c1  c2  check
permutation  w3  p3  r1  p1  r2  c3  w2  p2  c2  c1  check
permutation  w3  p3  r1  p1  r2  c3  w2  c1  p2  c2  check
permutation  w3  p3  r1  p1  r2  c3  c1  w2  p2  c2  check
permutation  w3  p3  r1  p1  w2  r2  p2  c3  c1  c2  check
permutation  w3  p3  r1  p1  w2  r2  p2  c3  c2  c1  check
permutation  w3  p3  r1  p1  w2  r2  c3  p2  c1  c2  check
permutation  w3  p3  r1  p1  w2  r2  c3  p2  c2  c1  check
permutation  w3  p3  r1  p1  w2  r2  c3  c1  p2  c2  check
permutation  w3  p3  r1  p1  w2  c3  r2  p2  c1  c2  check
permutation  w3  p3  r1  p1  w2  c3  r2  p2  c2  c1  check
permutation  w3  p3  r1  p1  w2  c3  r2  c1  p2  c2  check
permutation  w3  p3  r1  p1  w2  c3  c1  r2  p2  c2  check
permutation  w3  p3  r1  p1  c3  r2  w2  p2  c1  c2  check
permutation  w3  p3  r1  p1  c3  r2  w2  p2  c2  c1  check
permutation  w3  p3  r1  p1  c3  r2  w2  c1  p2  c2  check
permutation  w3  p3  r1  p1  c3  r2  c1  w2  p2  c2  check
permutation  w3  p3  r1  p1  c3  w2  r2  p2  c1  c2  check
permutation  w3  p3  r1  p1  c3  w2  r2  p2  c2  c1  check
permutation  w3  p3  r1  p1  c3  w2  r2  c1  p2  c2  check
permutation  w3  p3  r1  p1  c3  w2  c1  r2  p2  c2  check
permutation  w3  p3  r1  p1  c3  c1  r2  w2  p2  c2  check
permutation  w3  p3  r1  p1  c3  c1  w2  r2  p2  c2  check
permutation  w3  p3  r1  c3  r2  w2  p1  p2  c1  c2  check
permutation  w3  p3  r1  c3  r2  w2  p1  p2  c2  c1  check
permutation  w3  p3  r1  c3  r2  w2  p1  c1  p2  c2  check
permutation  w3  p3  r1  c3  r2  w2  p2  p1  c1  c2  check
permutation  w3  p3  r1  c3  r2  w2  p2  p1  c2  c1  check
permutation  w3  p3  r1  c3  r2  w2  p2  c2  p1  c1  check
permutation  w3  p3  r1  c3  r2  p1  w2  p2  c1  c2  check
permutation  w3  p3  r1  c3  r2  p1  w2  p2  c2  c1  check
permutation  w3  p3  r1  c3  r2  p1  w2  c1  p2  c2  check
permutation  w3  p3  r1  c3  r2  p1  c1  w2  p2  c2  check
permutation  w3  p3  r1  c3  w2  r2  p1  p2  c1  c2  check
permutation  w3  p3  r1  c3  w2  r2  p1  p2  c2  c1  check
permutation  w3  p3  r1  c3  w2  r2  p1  c1  p2  c2  check
permutation  w3  p3  r1  c3  w2  r2  p2  p1  c1  c2  check
permutation  w3  p3  r1  c3  w2  r2  p2  p1  c2  c1  check
permutation  w3  p3  r1  c3  w2  r2  p2  c2  p1  c1  check
permutation  w3  p3  r1  c3  w2  p1  r2  p2  c1  c2  check
permutation  w3  p3  r1  c3  w2  p1  r2  p2  c2  c1  check
permutation  w3  p3  r1  c3  w2  p1  r2  c1  p2  c2  check
permutation  w3  p3  r1  c3  w2  p1  c1  r2  p2  c2  check
permutation  w3  p3  r1  c3  p1  r2  w2  p2  c1  c2  check
permutation  w3  p3  r1  c3  p1  r2  w2  p2  c2  c1  check
permutation  w3  p3  r1  c3  p1  r2  w2  c1  p2  c2  check
permutation  w3  p3  r1  c3  p1  r2  c1  w2  p2  c2  check
permutation  w3  p3  r1  c3  p1  w2  r2  p2  c1  c2  check
permutation  w3  p3  r1  c3  p1  w2  r2  p2  c2  c1  check
permutation  w3  p3  r1  c3  p1  w2  r2  c1  p2  c2  check
permutation  w3  p3  r1  c3  p1  w2  c1  r2  p2  c2  check
permutation  w3  p3  r1  c3  p1  c1  r2  w2  p2  c2  check
permutation  w3  p3  r1  c3  p1  c1  w2  r2  p2  c2  check
permutation  w3  p3  r2  r1  w2  p1  p2  c3  c1  c2  check
permutation  w3  p3  r2  r1  w2  p1  p2  c3  c2  c1  check
permutation  w3  p3  r2  r1  w2  p1  c3  p2  c1  c2  check
permutation  w3  p3  r2  r1  w2  p1  c3  p2  c2  c1  check
permutation  w3  p3  r2  r1  w2  p1  c3  c1  p2  c2  check
permutation  w3  p3  r2  r1  w2  p2  p1  c3  c1  c2  check
permutation  w3  p3  r2  r1  w2  p2  p1  c3  c2  c1  check
permutation  w3  p3  r2  r1  w2  p2  c3  p1  c1  c2  check
permutation  w3  p3  r2  r1  w2  p2  c3  p1  c2  c1  check
permutation  w3  p3  r2  r1  w2  p2  c3  c2  p1  c1  check
permutation  w3  p3  r2  r1  w2  c3  p1  p2  c1  c2  check
permutation  w3  p3  r2  r1  w2  c3  p1  p2  c2  c1  check
permutation  w3  p3  r2  r1  w2  c3  p1  c1  p2  c2  check
permutation  w3  p3  r2  r1  w2  c3  p2  p1  c1  c2  check
permutation  w3  p3  r2  r1  w2  c3  p2  p1  c2  c1  check
permutation  w3  p3  r2  r1  w2  c3  p2  c2  p1  c1  check
permutation  w3  p3  r2  r1  p1  w2  p2  c3  c1  c2  check
permutation  w3  p3  r2  r1  p1  w2  p2  c3  c2  c1  check
permutation  w3  p3  r2  r1  p1  w2  c3  p2  c1  c2  check
permutation  w3  p3  r2  r1  p1  w2  c3  p2  c2  c1  check
permutation  w3  p3  r2  r1  p1  w2  c3  c1  p2  c2  check
permutation  w3  p3  r2  r1  p1  c3  w2  p2  c1  c2  check
permutation  w3  p3  r2  r1  p1  c3  w2  p2  c2  c1  check
permutation  w3  p3  r2  r1  p1  c3  w2  c1  p2  c2  check
permutation  w3  p3  r2  r1  p1  c3  c1  w2  p2  c2  check
permutation  w3  p3  r2  r1  c3  w2  p1  p2  c1  c2  check
permutation  w3  p3  r2  r1  c3  w2  p1  p2  c2  c1  check
permutation  w3  p3  r2  r1  c3  w2  p1  c1  p2  c2  check
permutation  w3  p3  r2  r1  c3  w2  p2  p1  c1  c2  check
permutation  w3  p3  r2  r1  c3  w2  p2  p1  c2  c1  check
permutation  w3  p3  r2  r1  c3  w2  p2  c2  p1  c1  check
permutation  w3  p3  r2  r1  c3  p1  w2  p2  c1  c2  check
permutation  w3  p3  r2  r1  c3  p1  w2  p2  c2  c1  check
permutation  w3  p3  r2  r1  c3  p1  w2  c1  p2  c2  check
permutation  w3  p3  r2  r1  c3  p1  c1  w2  p2  c2  check
permutation  w3  p3  r2  c3  r1  w2  p1  p2  c1  c2  check
permutation  w3  p3  r2  c3  r1  w2  p1  p2  c2  c1  check
permutation  w3  p3  r2  c3  r1  w2  p1  c1  p2  c2  check
permutation  w3  p3  r2  c3  r1  w2  p2  p1  c1  c2  check
permutation  w3  p3  r2  c3  r1  w2  p2  p1  c2  c1  check
permutation  w3  p3  r2  c3  r1  w2  p2  c2  p1  c1  check
permutation  w3  p3  r2  c3  r1  p1  w2  p2  c1  c2  check
permutation  w3  p3  r2  c3  r1  p1  w2  p2  c2  c1  check
permutation  w3  p3  r2  c3  r1  p1  w2  c1  p2  c2  check
permutation  w3  p3  r2  c3  r1  p1  c1  w2  p2  c2  check
permutation  w3  p3  c3  r1  r2  w2  p1  p2  c1  c2  check
permutation  w3  p3  c3  r1  r2  w2  p1  p2  c2  c1  check
permutation  w3  p3  c3  r1  r2  w2  p1  c1  p2  c2  check
permutation  w3  p3  c3  r1  r2  w2  p2  p1  c1  c2  check
permutation  w3  p3  c3  r1  r2  w2  p2  p1  c2  c1  check
permutation  w3  p3  c3  r1  r2  w2  p2  c2  p1  c1  check
permutation  w3  p3  c3  r1  r2  p1  w2  p2  c1  c2  check
permutation  w3  p3  c3  r1  r2  p1  w2  p2  c2  c1  check
permutation  w3  p3  c3  r1  r2  p1  w2  c1  p2  c2  check
permutation  w3  p3  c3  r1  r2  p1  c1  w2  p2  c2  check
permutation  w3  p3  c3  r1  w2  r2  p1  p2  c1  c2  check
permutation  w3  p3  c3  r1  w2  r2  p1  p2  c2  c1  check
permutation  w3  p3  c3  r1  w2  r2  p1  c1  p2  c2  check
permutation  w3  p3  c3  r1  w2  r2  p2  p1  c1  c2  check
permutation  w3  p3  c3  r1  w2  r2  p2  p1  c2  c1  check
permutation  w3  p3  c3  r1  w2  r2  p2  c2  p1  c1  check
permutation  w3  p3  c3  r1  w2  p1  r2  p2  c1  c2  check
permutation  w3  p3  c3  r1  w2  p1  r2  p2  c2  c1  check
permutation  w3  p3  c3  r1  w2  p1  r2  c1  p2  c2  check
permutation  w3  p3  c3  r1  w2  p1  c1  r2  p2  c2  check
permutation  w3  p3  c3  r1  p1  r2  w2  p2  c1  c2  check
permutation  w3  p3  c3  r1  p1  r2  w2  p2  c2  c1  check
permutation  w3  p3  c3  r1  p1  r2  w2  c1  p2  c2  check
permutation  w3  p3  c3  r1  p1  r2  c1  w2  p2  c2  check
permutation  w3  p3  c3  r1  p1  w2  r2  p2  c1  c2  check
permutation  w3  p3  c3  r1  p1  w2  r2  p2  c2  c1  check
permutation  w3  p3  c3  r1  p1  w2  r2  c1  p2  c2  check
permutation  w3  p3  c3  r1  p1  w2  c1  r2  p2  c2  check
permutation  w3  p3  c3  r1  p1  c1  r2  w2  p2  c2  check
permutation  w3  p3  c3  r1  p1  c1  w2  r2  p2  c2  check
permutation  w3  p3  c3  r2  r1  w2  p1  p2  c1  c2  check
permutation  w3  p3  c3  r2  r1  w2  p1  p2  c2  c1  check
permutation  w3  p3  c3  r2  r1  w2  p1  c1  p2  c2  check
permutation  w3  p3  c3  r2  r1  w2  p2  p1  c1  c2  check
permutation  w3  p3  c3  r2  r1  w2  p2  p1  c2  c1  check
permutation  w3  p3  c3  r2  r1  w2  p2  c2  p1  c1  check
permutation  w3  p3  c3  r2  r1  p1  w2  p2  c1  c2  check
permutation  w3  p3  c3  r2  r1  p1  w2  p2  c2  c1  check
permutation  w3  p3  c3  r2  r1  p1  w2  c1  p2  c2  check
permutation  w3  p3  c3  r2  r1  p1  c1  w2  p2  c2  check
