/* test_tls.c
 *
 * Copyright (C) 2006-2025 wolfSSL Inc.
 *
 * This file is part of wolfSSL.
 *
 * wolfSSL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * wolfSSL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1335, USA
 */

#include <tests/unit.h>

#ifdef NO_INLINE
    #include <wolfssl/wolfcrypt/misc.h>
#else
    #define WOLFSSL_MISC_INCLUDED
    #include <wolfcrypt/src/misc.c>
#endif

#include <tests/utils.h>
#include <tests/api/test_tls.h>


int test_utils_memio_move_message(void)
{
    EXPECT_DECLS;
#if defined(HAVE_MANUAL_MEMIO_TESTS_DEPENDENCIES)
    WOLFSSL_CTX *ctx_c = NULL, *ctx_s = NULL;
    WOLFSSL *ssl_c = NULL, *ssl_s = NULL;
    struct test_memio_ctx test_ctx;

    XMEMSET(&test_ctx, 0, sizeof(test_ctx));

    ExpectIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
            wolfTLS_client_method, wolfTLS_server_method), 0);
    wolfSSL_set_verify(ssl_s, WOLFSSL_VERIFY_PEER, NULL);
    ExpectIntEQ(wolfSSL_clear_group_messages(ssl_s), 1);
    /* start handshake, send first ClientHello */
    ExpectIntEQ(wolfSSL_connect(ssl_c), -1);
    ExpectIntEQ(wolfSSL_get_error(ssl_c, -1), WOLFSSL_ERROR_WANT_READ);
    /* send server's flight */
    ExpectIntEQ(wolfSSL_accept(ssl_s), -1);
    ExpectIntEQ(wolfSSL_get_error(ssl_s, -1), WOLFSSL_ERROR_WANT_READ);
    /* Move messages around but they should be the same at the end */
    ExpectIntEQ(test_memio_move_message(&test_ctx, 1, 1, 2), 0);
    ExpectIntEQ(test_memio_move_message(&test_ctx, 1, 2, 1), 0);
    ExpectIntEQ(test_memio_move_message(&test_ctx, 1, 1, 3), 0);
    ExpectIntEQ(test_memio_move_message(&test_ctx, 1, 3, 1), 0);
    ExpectIntEQ(test_memio_move_message(&test_ctx, 1, 0, 2), 0);
    ExpectIntEQ(test_memio_move_message(&test_ctx, 1, 2, 0), 0);
    ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);

    wolfSSL_free(ssl_c);
    wolfSSL_free(ssl_s);
    wolfSSL_CTX_free(ctx_c);
    wolfSSL_CTX_free(ctx_s);
#endif
    return EXPECT_RESULT();
}

int test_tls12_unexpected_ccs(void)
{
    EXPECT_DECLS;
#if defined(HAVE_MANUAL_MEMIO_TESTS_DEPENDENCIES) && !defined(WOLFSSL_NO_TLS12)
    const byte ccs[] = {
        0x14, /* ccs type */
        0x03, 0x03, /* version */
        0x00, 0x01, /* length */
        0x01, /* ccs value */
    };
    const byte badccs[] = {
        0x14, /* ccs type */
        0x03, 0x03, /* version */
        0x00, 0x01, /* length */
        0x99, /* wrong ccs value */
    };
    WOLFSSL_CTX *ctx_s = NULL;
    WOLFSSL *ssl_s = NULL;
    struct test_memio_ctx test_ctx;

    /* ccs in the wrong place */
    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    /* inject SH */
    ExpectIntEQ(test_memio_inject_message(&test_ctx, 0,
            (const char*)ccs, sizeof(ccs)), 0);
    ExpectIntEQ(test_memio_setup(&test_ctx, NULL, &ctx_s, NULL, &ssl_s,
                    NULL, wolfTLSv1_2_server_method), 0);
    ExpectIntEQ(wolfSSL_accept(ssl_s), WOLFSSL_FATAL_ERROR);
    ExpectIntEQ(wolfSSL_get_error(ssl_s, WOLFSSL_FATAL_ERROR),
            OUT_OF_ORDER_E);
    wolfSSL_free(ssl_s);
    wolfSSL_CTX_free(ctx_s);
    ctx_s = NULL;
    ssl_s = NULL;

    /* malformed ccs */
    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    ExpectIntEQ(test_memio_inject_message(&test_ctx, 0,
            (const char*)badccs, sizeof(badccs)), 0);
    ExpectIntEQ(test_memio_setup(&test_ctx, NULL, &ctx_s, NULL, &ssl_s,
                    NULL, wolfTLSv1_2_server_method), 0);
    ExpectIntEQ(wolfSSL_accept(ssl_s), WOLFSSL_FATAL_ERROR);
    ExpectIntEQ(wolfSSL_get_error(ssl_s, WOLFSSL_FATAL_ERROR),
            LENGTH_ERROR);
    wolfSSL_free(ssl_s);
    wolfSSL_CTX_free(ctx_s);
#endif
    return EXPECT_RESULT();
}

int test_tls13_unexpected_ccs(void)
{
    EXPECT_DECLS;
#if defined(HAVE_MANUAL_MEMIO_TESTS_DEPENDENCIES) && defined(WOLFSSL_TLS13)
    const byte ccs[] = {
        0x14, /* ccs type */
        0x03, 0x03, /* version */
        0x00, 0x01, /* length */
        0x01, /* ccs value */
    };
    const byte badccs[] = {
        0x14, /* ccs type */
        0x03, 0x03, /* version */
        0x00, 0x01, /* length */
        0x99, /* wrong ccs value */
    };
    const byte unexpectedAlert[] = {
        0x15, /* alert type */
        0x03, 0x03, /* version */
        0x00, 0x02, /* length */
        0x02, /* level: fatal */
        0x0a /* protocol version */
    };
    WOLFSSL_CTX *ctx_s = NULL;
    WOLFSSL *ssl_s = NULL;
    struct test_memio_ctx test_ctx;

    /* ccs can't appear before a CH */
    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    ExpectIntEQ(test_memio_inject_message(&test_ctx, 0,
            (const char*)ccs, sizeof(ccs)), 0);
    ExpectIntEQ(test_memio_setup(&test_ctx, NULL, &ctx_s, NULL, &ssl_s,
                    NULL, wolfTLSv1_3_server_method), 0);
    ExpectIntEQ(wolfSSL_accept(ssl_s), WOLFSSL_FATAL_ERROR);
    ExpectIntEQ(wolfSSL_get_error(ssl_s, WOLFSSL_FATAL_ERROR),
            UNKNOWN_RECORD_TYPE);
    ExpectIntEQ(test_ctx.c_len, sizeof(unexpectedAlert));
    ExpectBufEQ(test_ctx.c_buff, unexpectedAlert, sizeof(unexpectedAlert));
    wolfSSL_free(ssl_s);
    wolfSSL_CTX_free(ctx_s);
    ctx_s = NULL;
    ssl_s = NULL;

    /* malformed ccs */
    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    ExpectIntEQ(test_memio_inject_message(&test_ctx, 0,
            (const char*)badccs, sizeof(badccs)), 0);
    ExpectIntEQ(test_memio_setup(&test_ctx, NULL, &ctx_s, NULL, &ssl_s,
                    NULL, wolfTLSv1_3_server_method), 0);
    ExpectIntEQ(wolfSSL_accept(ssl_s), WOLFSSL_FATAL_ERROR);
    ExpectIntEQ(wolfSSL_get_error(ssl_s, WOLFSSL_FATAL_ERROR),
            UNKNOWN_RECORD_TYPE);
    ExpectIntEQ(test_ctx.c_len, sizeof(unexpectedAlert));
    ExpectBufEQ(test_ctx.c_buff, unexpectedAlert, sizeof(unexpectedAlert));
    wolfSSL_free(ssl_s);
    wolfSSL_CTX_free(ctx_s);
#endif
    return EXPECT_RESULT();
}
int test_tls12_curve_intersection(void) {
    EXPECT_DECLS;
#if defined(HAVE_MANUAL_MEMIO_TESTS_DEPENDENCIES) && \
    !defined(WOLFSSL_NO_TLS12) && defined(HAVE_ECC) && \
    defined(HAVE_CURVE25519)
    WOLFSSL_CTX *ctx_c = NULL, *ctx_s = NULL;
    WOLFSSL *ssl_c = NULL, *ssl_s = NULL;
    struct test_memio_ctx test_ctx;
    int ret;
    const char* curve_name;
    int test1[] = {WOLFSSL_ECC_SECP256R1};
    int test2[] = {WOLFSSL_ECC_SECP384R1};
    int test3[] = {WOLFSSL_ECC_SECP256R1, WOLFSSL_ECC_SECP384R1};
    int test4[] = {WOLFSSL_ECC_SECP384R1, WOLFSSL_ECC_SECP256R1};
    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    ExpectIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
                    wolfTLSv1_2_client_method, wolfTLSv1_2_server_method), 0);
    ExpectIntEQ(wolfSSL_set_groups(ssl_c,
                    test1, 1), WOLFSSL_SUCCESS);
    ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);

    /* Fix: Get curve name and compare with string comparison or use curve
     * ID function */
    curve_name = wolfSSL_get_curve_name(ssl_s);
    /* or use appropriate string comparison */
    ExpectStrEQ(curve_name, "SECP256R1");
    curve_name = wolfSSL_get_curve_name(ssl_c);
    ExpectStrEQ(curve_name, "SECP256R1");

    wolfSSL_free(ssl_c);
    wolfSSL_free(ssl_s);
    wolfSSL_CTX_free(ctx_c);
    wolfSSL_CTX_free(ctx_s);
    ssl_c = NULL;
    ssl_s = NULL;
    ctx_c = NULL;
    ctx_s = NULL;

    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    ExpectIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
                    wolfTLSv1_2_client_method, wolfTLSv1_2_server_method), 0);
    ExpectIntEQ(wolfSSL_set_groups(ssl_c,
                   test2, 1), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_set_groups(ssl_s,
                    test1, 1), WOLFSSL_SUCCESS);
    ExpectIntNE(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
    ret = wolfSSL_get_error(ssl_s, WOLFSSL_FATAL_ERROR);

    /* Fix: Use proper constant or define HANDSHAKE_FAILURE */
    ExpectTrue(ret == WC_NO_ERR_TRACE(ECC_CURVE_ERROR));

    wolfSSL_free(ssl_c);
    wolfSSL_free(ssl_s);
    wolfSSL_CTX_free(ctx_c);
    wolfSSL_CTX_free(ctx_s);
    ssl_c = NULL;
    ssl_s = NULL;
    ctx_c = NULL;
    ctx_s = NULL;

    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    ExpectIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
                    wolfTLSv1_2_client_method, wolfTLSv1_2_server_method), 0);
    ExpectIntEQ(wolfSSL_set_groups(ssl_c,
                    test3, 2),
                    WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_set_groups(ssl_s,
                    test4, 2),
                    WOLFSSL_SUCCESS);
    ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);

    curve_name = wolfSSL_get_curve_name(ssl_s);
    ExpectStrEQ(curve_name, "SECP256R1");
    curve_name = wolfSSL_get_curve_name(ssl_c);
    ExpectStrEQ(curve_name, "SECP256R1");

    wolfSSL_free(ssl_c);
    wolfSSL_free(ssl_s);
    wolfSSL_CTX_free(ctx_c);
    wolfSSL_CTX_free(ctx_s);
#endif
    return EXPECT_RESULT();
}

int test_tls13_curve_intersection(void) {
    EXPECT_DECLS;
#if defined(HAVE_MANUAL_MEMIO_TESTS_DEPENDENCIES) && \
    defined(WOLFSSL_TLS13) && defined(HAVE_ECC) && defined(HAVE_CURVE25519)
    WOLFSSL_CTX *ctx_c = NULL, *ctx_s = NULL;
    WOLFSSL *ssl_c = NULL, *ssl_s = NULL;
    struct test_memio_ctx test_ctx;
    const char* curve_name;
    int test1[] ={WOLFSSL_ECC_SECP256R1};

    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    ExpectIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
                    wolfTLSv1_3_client_method, wolfTLSv1_3_server_method), 0);
    ExpectIntEQ(wolfSSL_set_groups(ssl_c,
                    test1, 1), WOLFSSL_SUCCESS);
    ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);

    curve_name = wolfSSL_get_curve_name(ssl_s);
    ExpectStrEQ(curve_name, "SECP256R1");
    curve_name = wolfSSL_get_curve_name(ssl_c);
    ExpectStrEQ(curve_name, "SECP256R1");

    wolfSSL_free(ssl_c);
    wolfSSL_free(ssl_s);
    wolfSSL_CTX_free(ctx_c);
    wolfSSL_CTX_free(ctx_s);
#endif
    return EXPECT_RESULT();
}


int test_tls_certreq_order(void)
{
    EXPECT_DECLS;
#if defined(HAVE_MANUAL_MEMIO_TESTS_DEPENDENCIES) && \
    !defined(WOLFSSL_NO_TLS12) && defined(HAVE_AESGCM) && \
    defined(WOLFSSL_AES_256) && defined(WOLFSSL_SHA384) && !defined(NO_RSA) && \
    defined(HAVE_ECC)
    /* This test checks that a certificate request message
     * received before server certificate message is properly detected.
     */
    WOLFSSL_CTX *ctx_c = NULL, *ctx_s = NULL;
    WOLFSSL *ssl_c = NULL, *ssl_s = NULL;
    struct test_memio_ctx test_ctx;
    int i = 0;
    const char* msg = NULL;
    int msgSz = 0;
    int certIdx = 0;
    int certReqIdx = 0;

    XMEMSET(&test_ctx, 0, sizeof(test_ctx));

    ExpectIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
                    wolfTLSv1_2_client_method, wolfTLSv1_2_server_method), 0);
    wolfSSL_set_verify(ssl_s, WOLFSSL_VERIFY_PEER, NULL);
    ExpectIntEQ(wolfSSL_clear_group_messages(ssl_s), 1);

    /* start handshake, send first ClientHello */
    ExpectIntEQ(wolfSSL_connect(ssl_c), -1);
    ExpectIntEQ(wolfSSL_get_error(ssl_c, -1), WOLFSSL_ERROR_WANT_READ);
    /* send server's flight */
    ExpectIntEQ(wolfSSL_accept(ssl_s), -1);
    ExpectIntEQ(wolfSSL_get_error(ssl_s, -1), WOLFSSL_ERROR_WANT_READ);
    for (i = 0; test_memio_get_message(&test_ctx, 1, &msg, &msgSz, i) == 0; i++) {
        if (msg[5] == 11) /* cert */
            certIdx = i;
        if (msg[5] == 13) /* certreq */
            certReqIdx = i;
    }
    ExpectIntNE(certIdx, 0);
    ExpectIntNE(certReqIdx, 0);
    ExpectIntEQ(test_memio_move_message(&test_ctx, 1, certReqIdx, certIdx), 0);
    ExpectIntEQ(wolfSSL_connect(ssl_c), -1);
    ExpectIntEQ(wolfSSL_get_error(ssl_c, -1), OUT_OF_ORDER_E);

    wolfSSL_free(ssl_c);
    wolfSSL_free(ssl_s);
    wolfSSL_CTX_free(ctx_c);
    wolfSSL_CTX_free(ctx_s);
#endif
    return EXPECT_RESULT();
}

#if !defined(WOLFSSL_NO_TLS12) && !defined(NO_RSA) && defined(HAVE_ECC) && \
    !defined(NO_WOLFSSL_SERVER)
/* Called when writing. */
static int CsSend(WOLFSSL* ssl, char* buf, int sz, void* ctx)
{
    (void)ssl;
    (void)buf;
    (void)ctx;

    return sz;
}
/* Called when reading. */
static int CsRecv(WOLFSSL* ssl, char* buf, int sz, void* ctx)
{
    WOLFSSL_BUFFER_INFO* msg = (WOLFSSL_BUFFER_INFO*)ctx;
    int len = (int)msg->length;

    (void)ssl;
    (void)sz;

    /* Pass back as much of message as will fit in buffer. */
    if (len > sz)
        len = sz;
    XMEMCPY(buf, msg->buffer, len);
    /* Move over returned data. */
    msg->buffer += len;
    msg->length -= len;

    /* Amount actually copied. */
    return len;
}
#endif

int test_tls12_bad_cv_sig_alg(void)
{
    EXPECT_DECLS;
#if !defined(WOLFSSL_NO_TLS12) && !defined(NO_RSA) && defined(HAVE_ECC) && \
    !defined(NO_WOLFSSL_SERVER)
    byte clientMsgs[] = {
        /* Client Hello */
        0x16, 0x03, 0x03, 0x00, 0xe7,
        0x01, 0x00, 0x00, 0xe3, 0x03, 0x03, 0x65, 0x27,
        0x41, 0xdf, 0xd9, 0x17, 0xdb, 0x02, 0x5c, 0x2e,
        0xf8, 0x4b, 0x77, 0x86, 0x5a, 0x20, 0x57, 0x7f,
        0xc0, 0xe7, 0xef, 0x8f, 0x56, 0xef, 0xfa, 0x71,
        0x36, 0xec, 0x55, 0x1d, 0x4e, 0xa2, 0x00, 0x00,
        0x64, 0xc0, 0x2c, 0xc0, 0x2b, 0xc0, 0x30, 0xc0,
        0x2f, 0x00, 0x9f, 0x00, 0x9e, 0x00, 0xab, 0x00,
        0x34, 0x00, 0xa7, 0x00, 0xaa, 0xcc, 0xa9, 0xcc,
        0xa8, 0xcc, 0xaa, 0xc0, 0x27, 0xc0, 0x23, 0xc0,
        0x28, 0xc0, 0x24, 0xc0, 0x0a, 0xc0, 0x09, 0xc0,
        0x07, 0xc0, 0x14, 0xc0, 0x13, 0xc0, 0x11, 0xc0,
        0xac, 0xc0, 0xae, 0xc0, 0xaf, 0x00, 0x6b, 0x00,
        0x67, 0x00, 0x39, 0x00, 0x33, 0xcc, 0x14, 0xcc,
        0x13, 0xcc, 0x15, 0xc0, 0x06, 0x00, 0xb3, 0x00,
        0xb2, 0xc0, 0xa6, 0xc0, 0xa7, 0xcc, 0xab, 0xcc,
        0xac, 0xcc, 0xad, 0xc0, 0x37, 0xd0, 0x01, 0x00,
        0xb5, 0xc0, 0x3a, 0x00, 0xb4, 0x00, 0x45, 0x00,
        0x88, 0x00, 0xbe, 0x00, 0xc4, 0x01, 0x00, 0x00,
        0x56, 0x00, 0x0d, 0x00, 0x20, 0x00, 0x1e, 0x06,
        0x03, 0x05, 0x03, 0x04, 0x03, 0x08, 0x07, 0x08,
        0x08, 0x08, 0x06, 0x08, 0x0b, 0x08, 0x05, 0x08,
        0x0a, 0x08, 0x04, 0x08, 0x09, 0x06, 0x01, 0x05,
        0x01, 0x04, 0x01, 0x03, 0x01, 0x00, 0x0b, 0x00,
        0x02, 0x01, 0x00, 0x00, 0x0a, 0x00, 0x1c, 0x00,
        0x1a, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x18, 0x00,
        0x1b, 0x00, 0x1e, 0x00, 0x17, 0x00, 0x16, 0x00,
        0x1a, 0x00, 0x1d, 0x00, 0x15, 0x00, 0x14, 0x01,
        0x01, 0x01, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00,
        0x23, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00,
        /* Certificate */
        0x16, 0x03, 0x03, 0x05, 0x2b,
        0x0b, 0x00, 0x05, 0x27, 0x00, 0x05, 0x24, 0x00,
        0x05, 0x21, 0x30, 0x82, 0x05, 0x1d, 0x30, 0x82,
        0x04, 0x05, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02,
        0x14, 0x4f, 0x0d, 0x8c, 0xc5, 0xfa, 0xee, 0xa2,
        0x9b, 0xb7, 0x35, 0x9e, 0xe9, 0x4a, 0x17, 0x99,
        0xf0, 0xcc, 0x23, 0xf2, 0xec, 0x30, 0x0d, 0x06,
        0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
        0x01, 0x0b, 0x05, 0x00, 0x30, 0x81, 0x9e, 0x31,
        0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
        0x13, 0x02, 0x55, 0x53, 0x31, 0x10, 0x30, 0x0e,
        0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x07, 0x4d,
        0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x31, 0x10,
        0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c,
        0x07, 0x42, 0x6f, 0x7a, 0x65, 0x6d, 0x61, 0x6e,
        0x31, 0x15, 0x30, 0x13, 0x06, 0x03, 0x55, 0x04,
        0x0a, 0x0c, 0x0c, 0x77, 0x6f, 0x6c, 0x66, 0x53,
        0x53, 0x4c, 0x5f, 0x32, 0x30, 0x34, 0x38, 0x31,
        0x19, 0x30, 0x17, 0x06, 0x03, 0x55, 0x04, 0x0b,
        0x0c, 0x10, 0x50, 0x72, 0x6f, 0x67, 0x72, 0x61,
        0x6d, 0x6d, 0x69, 0x6e, 0x67, 0x2d, 0x32, 0x30,
        0x34, 0x38, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03,
        0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, 0x77, 0x77,
        0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c,
        0x2e, 0x63, 0x6f, 0x6d, 0x31, 0x1f, 0x30, 0x1d,
        0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
        0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6e, 0x66,
        0x6f, 0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73,
        0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x1e, 0x17,
        0x0d, 0x32, 0x34, 0x31, 0x32, 0x31, 0x38, 0x32,
        0x31, 0x32, 0x35, 0x32, 0x39, 0x5a, 0x17, 0x0d,
        0x32, 0x37, 0x30, 0x39, 0x31, 0x34, 0x32, 0x31,
        0x32, 0x35, 0x32, 0x39, 0x5a, 0x30, 0x81, 0x9e,
        0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04,
        0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x10, 0x30,
        0x0e, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x07,
        0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x31,
        0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07,
        0x0c, 0x07, 0x42, 0x6f, 0x7a, 0x65, 0x6d, 0x61,
        0x6e, 0x31, 0x15, 0x30, 0x13, 0x06, 0x03, 0x55,
        0x04, 0x0a, 0x0c, 0x0c, 0x77, 0x6f, 0x6c, 0x66,
        0x53, 0x53, 0x4c, 0x5f, 0x32, 0x30, 0x34, 0x38,
        0x31, 0x19, 0x30, 0x17, 0x06, 0x03, 0x55, 0x04,
        0x0b, 0x0c, 0x10, 0x50, 0x72, 0x6f, 0x67, 0x72,
        0x61, 0x6d, 0x6d, 0x69, 0x6e, 0x67, 0x2d, 0x32,
        0x30, 0x34, 0x38, 0x31, 0x18, 0x30, 0x16, 0x06,
        0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, 0x77,
        0x77, 0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73,
        0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x31, 0x1f, 0x30,
        0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
        0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6e,
        0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73,
        0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x82,
        0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
        0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05,
        0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82,
        0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xc3,
        0x03, 0xd1, 0x2b, 0xfe, 0x39, 0xa4, 0x32, 0x45,
        0x3b, 0x53, 0xc8, 0x84, 0x2b, 0x2a, 0x7c, 0x74,
        0x9a, 0xbd, 0xaa, 0x2a, 0x52, 0x07, 0x47, 0xd6,
        0xa6, 0x36, 0xb2, 0x07, 0x32, 0x8e, 0xd0, 0xba,
        0x69, 0x7b, 0xc6, 0xc3, 0x44, 0x9e, 0xd4, 0x81,
        0x48, 0xfd, 0x2d, 0x68, 0xa2, 0x8b, 0x67, 0xbb,
        0xa1, 0x75, 0xc8, 0x36, 0x2c, 0x4a, 0xd2, 0x1b,
        0xf7, 0x8b, 0xba, 0xcf, 0x0d, 0xf9, 0xef, 0xec,
        0xf1, 0x81, 0x1e, 0x7b, 0x9b, 0x03, 0x47, 0x9a,
        0xbf, 0x65, 0xcc, 0x7f, 0x65, 0x24, 0x69, 0xa6,
        0xe8, 0x14, 0x89, 0x5b, 0xe4, 0x34, 0xf7, 0xc5,
        0xb0, 0x14, 0x93, 0xf5, 0x67, 0x7b, 0x3a, 0x7a,
        0x78, 0xe1, 0x01, 0x56, 0x56, 0x91, 0xa6, 0x13,
        0x42, 0x8d, 0xd2, 0x3c, 0x40, 0x9c, 0x4c, 0xef,
        0xd1, 0x86, 0xdf, 0x37, 0x51, 0x1b, 0x0c, 0xa1,
        0x3b, 0xf5, 0xf1, 0xa3, 0x4a, 0x35, 0xe4, 0xe1,
        0xce, 0x96, 0xdf, 0x1b, 0x7e, 0xbf, 0x4e, 0x97,
        0xd0, 0x10, 0xe8, 0xa8, 0x08, 0x30, 0x81, 0xaf,
        0x20, 0x0b, 0x43, 0x14, 0xc5, 0x74, 0x67, 0xb4,
        0x32, 0x82, 0x6f, 0x8d, 0x86, 0xc2, 0x88, 0x40,
        0x99, 0x36, 0x83, 0xba, 0x1e, 0x40, 0x72, 0x22,
        0x17, 0xd7, 0x52, 0x65, 0x24, 0x73, 0xb0, 0xce,
        0xef, 0x19, 0xcd, 0xae, 0xff, 0x78, 0x6c, 0x7b,
        0xc0, 0x12, 0x03, 0xd4, 0x4e, 0x72, 0x0d, 0x50,
        0x6d, 0x3b, 0xa3, 0x3b, 0xa3, 0x99, 0x5e, 0x9d,
        0xc8, 0xd9, 0x0c, 0x85, 0xb3, 0xd9, 0x8a, 0xd9,
        0x54, 0x26, 0xdb, 0x6d, 0xfa, 0xac, 0xbb, 0xff,
        0x25, 0x4c, 0xc4, 0xd1, 0x79, 0xf4, 0x71, 0xd3,
        0x86, 0x40, 0x18, 0x13, 0xb0, 0x63, 0xb5, 0x72,
        0x4e, 0x30, 0xc4, 0x97, 0x84, 0x86, 0x2d, 0x56,
        0x2f, 0xd7, 0x15, 0xf7, 0x7f, 0xc0, 0xae, 0xf5,
        0xfc, 0x5b, 0xe5, 0xfb, 0xa1, 0xba, 0xd3, 0x02,
        0x03, 0x01, 0x00, 0x01, 0xa3, 0x82, 0x01, 0x4f,
        0x30, 0x82, 0x01, 0x4b, 0x30, 0x1d, 0x06, 0x03,
        0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x33,
        0xd8, 0x45, 0x66, 0xd7, 0x68, 0x87, 0x18, 0x7e,
        0x54, 0x0d, 0x70, 0x27, 0x91, 0xc7, 0x26, 0xd7,
        0x85, 0x65, 0xc0, 0x30, 0x81, 0xde, 0x06, 0x03,
        0x55, 0x1d, 0x23, 0x04, 0x81, 0xd6, 0x30, 0x81,
        0xd3, 0x80, 0x14, 0x33, 0xd8, 0x45, 0x66, 0xd7,
        0x68, 0x87, 0x18, 0x7e, 0x54, 0x0d, 0x70, 0x27,
        0x91, 0xc7, 0x26, 0xd7, 0x85, 0x65, 0xc0, 0xa1,
        0x81, 0xa4, 0xa4, 0x81, 0xa1, 0x30, 0x81, 0x9e,
        0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04,
        0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x10, 0x30,
        0x0e, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x07,
        0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x31,
        0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07,
        0x0c, 0x07, 0x42, 0x6f, 0x7a, 0x65, 0x6d, 0x61,
        0x6e, 0x31, 0x15, 0x30, 0x13, 0x06, 0x03, 0x55,
        0x04, 0x0a, 0x0c, 0x0c, 0x77, 0x6f, 0x6c, 0x66,
        0x53, 0x53, 0x4c, 0x5f, 0x32, 0x30, 0x34, 0x38,
        0x31, 0x19, 0x30, 0x17, 0x06, 0x03, 0x55, 0x04,
        0x0b, 0x0c, 0x10, 0x50, 0x72, 0x6f, 0x67, 0x72,
        0x61, 0x6d, 0x6d, 0x69, 0x6e, 0x67, 0x2d, 0x32,
        0x30, 0x34, 0x38, 0x31, 0x18, 0x30, 0x16, 0x06,
        0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, 0x77,
        0x77, 0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73,
        0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x31, 0x1f, 0x30,
        0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
        0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6e,
        0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73,
        0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x82, 0x14,
        0x4f, 0x0d, 0x8c, 0xc5, 0xfa, 0xee, 0xa2, 0x9b,
        0xb7, 0x35, 0x9e, 0xe9, 0x4a, 0x17, 0x99, 0xf0,
        0xcc, 0x23, 0xf2, 0xec, 0x30, 0x0c, 0x06, 0x03,
        0x55, 0x1d, 0x13, 0x04, 0x05, 0x30, 0x03, 0x01,
        0x01, 0xff, 0x30, 0x1c, 0x06, 0x03, 0x55, 0x1d,
        0x11, 0x04, 0x15, 0x30, 0x13, 0x82, 0x0b, 0x65,
        0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2e, 0x63,
        0x6f, 0x6d, 0x87, 0x04, 0x7f, 0x00, 0x00, 0x01,
        0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x04,
        0x16, 0x30, 0x14, 0x06, 0x08, 0x2b, 0x06, 0x01,
        0x05, 0x05, 0x07, 0x03, 0x01, 0x06, 0x08, 0x2b,
        0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02, 0x30,
        0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
        0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82,
        0x01, 0x01, 0x00, 0x46, 0xab, 0xe4, 0x6d, 0xae,
        0x49, 0x5b, 0x6a, 0x0b, 0xa9, 0x87, 0xe1, 0x95,
        0x32, 0xa6, 0xd7, 0xae, 0xde, 0x28, 0xdc, 0xc7,
        0x99, 0x68, 0xe2, 0x5f, 0xc9, 0x5a, 0x4c, 0x64,
        0xb8, 0xf5, 0x28, 0x42, 0x5a, 0xe8, 0x5c, 0x59,
        0x32, 0xfe, 0xd0, 0x1f, 0x0b, 0x55, 0x89, 0xdb,
        0x67, 0xe7, 0x78, 0xf3, 0x70, 0xcf, 0x18, 0x51,
        0x57, 0x8b, 0xf3, 0x2b, 0xa4, 0x66, 0x0b, 0xf6,
        0x03, 0x6e, 0x11, 0xac, 0x83, 0x52, 0x16, 0x7e,
        0xa2, 0x7c, 0x36, 0x77, 0xf6, 0xbb, 0x13, 0x19,
        0x40, 0x2c, 0xb8, 0x8c, 0xca, 0xd6, 0x7e, 0x79,
        0x7d, 0xf4, 0x14, 0x8d, 0xb5, 0xa4, 0x09, 0xf6,
        0x2d, 0x4c, 0xe7, 0xf9, 0xb8, 0x25, 0x41, 0x15,
        0x78, 0xf4, 0xca, 0x80, 0x41, 0xea, 0x3a, 0x05,
        0x08, 0xf6, 0xb5, 0x5b, 0xa1, 0x3b, 0x5b, 0x48,
        0xa8, 0x4b, 0x8c, 0x19, 0x8d, 0x6c, 0x87, 0x31,
        0x76, 0x74, 0x02, 0x16, 0x8b, 0xdd, 0x7f, 0xd1,
        0x11, 0x62, 0x27, 0x42, 0x39, 0xe0, 0x9a, 0x63,
        0x26, 0x31, 0x19, 0xce, 0x3d, 0x41, 0xd5, 0x24,
        0x47, 0x32, 0x0f, 0x76, 0xd6, 0x41, 0x37, 0x44,
        0xad, 0x73, 0xf1, 0xb8, 0xec, 0x2b, 0x6e, 0x9c,
        0x4f, 0x84, 0xc4, 0x4e, 0xd7, 0x92, 0x10, 0x7e,
        0x23, 0x32, 0xa0, 0x75, 0x6a, 0xe7, 0xfe, 0x55,
        0x95, 0x9f, 0x0a, 0xad, 0xdf, 0xf9, 0x2a, 0xa2,
        0x1a, 0x59, 0xd5, 0x82, 0x63, 0xd6, 0x5d, 0x7d,
        0x79, 0xf4, 0xa7, 0x2d, 0xdc, 0x8c, 0x04, 0xcd,
        0x98, 0xb0, 0x42, 0x0e, 0x84, 0xfa, 0x86, 0x50,
        0x10, 0x61, 0xac, 0x73, 0xcd, 0x79, 0x45, 0x30,
        0xe8, 0x42, 0xa1, 0x6a, 0xf6, 0x77, 0x55, 0xec,
        0x07, 0xdb, 0x52, 0x29, 0xca, 0x7a, 0xc8, 0xa2,
        0xda, 0xe9, 0xf5, 0x98, 0x33, 0x6a, 0xe8, 0xbc,
        0x89, 0xed, 0x01, 0xe2, 0xfe, 0x44, 0x86, 0x86,
        0x80, 0x39, 0xec,
        /* ClientKeyExchange */
        0x16, 0x03, 0x03, 0x00, 0x46,
        0x10, 0x00, 0x00, 0x42, 0x41, 0x04, 0xc5, 0xb9,
        0x0f, 0xbc, 0x84, 0xe6, 0x0c, 0x02, 0xa6, 0x8d,
        0x34, 0xa6, 0x3e, 0x1e, 0xb7, 0x88, 0xb8, 0x68,
        0x29, 0x2b, 0x85, 0x67, 0xe2, 0x62, 0x4d, 0xd9,
        0xa4, 0x38, 0xb3, 0xec, 0x33, 0xa1, 0xe5, 0xe1,
        0xae, 0xe9, 0x07, 0xd1, 0xea, 0x1b, 0xec, 0xa6,
        0xaf, 0x1f, 0x80, 0x87, 0x7c, 0x53, 0x80, 0x04,
        0xee, 0x20, 0xeb, 0x64, 0x0d, 0xa0, 0xf7, 0x62,
        0xb1, 0xcc, 0x73, 0x97, 0xf5, 0x80,
        /* CertificateVerify */
        0x16, 0x03, 0x03, 0x01, 0x08,
        /*                            0x04 - sha256, changed to 0x02 - sha1 */
        0x0f, 0x00, 0x01, 0x04, 0x08, 0x02, 0x01, 0x00,
        0x8b, 0x09, 0xa4, 0x58, 0x8d, 0x68, 0xd9, 0xc9,
        0xef, 0xe9, 0xa5, 0x98, 0x7f, 0xa3, 0xa9, 0x7b,
        0x56, 0xf7, 0xaa, 0x5f, 0x8f, 0x47, 0x7f, 0xd0,
        0x7b, 0xcf, 0x4f, 0x84, 0xe1, 0xa9, 0x0e, 0xa8,
        0x83, 0x19, 0xd8, 0xb3, 0x97, 0x23, 0x98, 0xc5,
        0x2b, 0x56, 0x82, 0x66, 0x94, 0xcc, 0xd7, 0x23,
        0xe6, 0x6e, 0x60, 0x83, 0x78, 0xfb, 0xaf, 0x8e,
        0x8b, 0xae, 0x1f, 0x3c, 0x34, 0x96, 0x3b, 0xd5,
        0x8d, 0x1e, 0xaf, 0x98, 0x1d, 0x27, 0x86, 0x97,
        0x42, 0xd4, 0xfc, 0x62, 0xbc, 0x43, 0x94, 0x98,
        0x19, 0x26, 0x87, 0xb0, 0x8c, 0xb5, 0x22, 0xa7,
        0x6a, 0x5e, 0x56, 0x73, 0x0a, 0x75, 0xc9, 0xb9,
        0x0e, 0xf7, 0x49, 0x4f, 0xa2, 0x0f, 0xfb, 0xdf,
        0x3e, 0xe4, 0xc8, 0x31, 0x26, 0xc5, 0x5c, 0x83,
        0x9f, 0x13, 0xcb, 0x4c, 0xdc, 0x21, 0xe6, 0x24,
        0x2d, 0xd3, 0xe8, 0x18, 0x04, 0xaf, 0x5c, 0x42,
        0x03, 0xa3, 0x0a, 0xb5, 0xfc, 0xb9, 0xbc, 0x8e,
        0xd3, 0xe0, 0x78, 0xdc, 0xef, 0xb9, 0x91, 0x9f,
        0x5b, 0xdc, 0xe3, 0x84, 0xd2, 0xca, 0x32, 0x33,
        0x00, 0x7c, 0x13, 0xd3, 0x2d, 0x85, 0x65, 0x00,
        0xc0, 0xb0, 0xde, 0x85, 0x37, 0x38, 0x18, 0xd2,
        0x81, 0xd4, 0x35, 0xeb, 0xf1, 0xfb, 0x9f, 0x6c,
        0x96, 0x95, 0xf5, 0xaa, 0xfd, 0x22, 0xca, 0x20,
        0xfd, 0x3b, 0xa9, 0xa7, 0xb6, 0x5a, 0x26, 0x02,
        0xb6, 0x0e, 0xdd, 0xaa, 0x0f, 0xa8, 0x96, 0x18,
        0xaa, 0xb1, 0x79, 0x9c, 0x17, 0xb0, 0x7e, 0xa7,
        0x4f, 0xc0, 0x98, 0x27, 0xbe, 0xac, 0x00, 0xda,
        0x3b, 0x2e, 0xd4, 0x11, 0x41, 0x54, 0x34, 0x53,
        0x5f, 0xc5, 0xcd, 0x72, 0xd7, 0x36, 0x04, 0xe1,
        0x7f, 0xcf, 0x1e, 0x01, 0x97, 0xec, 0xeb, 0xad,
        0x1c, 0xc6, 0x7f, 0x2d, 0x8c, 0x68, 0x29, 0xd1,
        0x93, 0x47, 0x59, 0xc0, 0xe2, 0x4a, 0x36, 0x6c
    };
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL* ssl = NULL;
    WOLFSSL_BUFFER_INFO msg;

    /* Set up wolfSSL context. */
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_server_method()));
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
        CERT_FILETYPE));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
        CERT_FILETYPE));
    if (EXPECT_SUCCESS()) {
        wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_NONE, NULL);
    }
    /* Read from 'msg'. */
    wolfSSL_SetIORecv(ctx, CsRecv);
    /* No where to send to - dummy sender. */
    wolfSSL_SetIOSend(ctx, CsSend);

    ExpectNotNull(ssl = wolfSSL_new(ctx));
    msg.buffer = clientMsgs;
    msg.length = (unsigned int)sizeof(clientMsgs);
    if (EXPECT_SUCCESS()) {
        wolfSSL_SetIOReadCtx(ssl, &msg);
    }
    /* Read all message  include CertificateVerify with invalid signature
     * algorithm. */
    ExpectIntEQ(wolfSSL_accept(ssl), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    /* Expect an invalid parameter error. */
    ExpectIntEQ(wolfSSL_get_error(ssl, WOLFSSL_FATAL_ERROR), -425);
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

